<?php

declare(strict_types = 1);

namespace JuicyCodes\Signer;

use Illuminate\Encryption\Encrypter as DefaultEncrypter;
use InvalidArgumentException;

class Encrypter
{
    protected DefaultEncrypter $encrypter;

    public function __construct(
        protected string $secret,
        protected string $cipher
    ) {
        $this->encrypter = new DefaultEncrypter($secret, $cipher);
    }

    public function encrypt(string $value): string
    {
        // Encrypt the given value & decode the encrypted payload
        $encrypt = $this->encrypter->encryptString($value);
        $payload = $this->decodeEncryptedPayload($encrypt);

        // Reorganize & re-encode the decrypted payload
        $encrypted = implode(".", [
            $payload->mac,
            $this->base64Encode(base64_decode($payload->iv)),
            $this->base64Encode(base64_decode($payload->value)),
        ]);

        return str_rot13($this->base64Encode($encrypted));
    }

    public function decrypt(string $encrypted): string
    {
        // Decode the given value & organize the encrypted payload
        $payload = $this->base64Decode(str_rot13($encrypted));
        [$mac, $iv, $value] = explode(".", $payload);

        // Re-encode the payload to supported format
        $iv    = base64_encode($this->base64Decode($iv));
        $value = base64_encode($this->base64Decode($value));

        // Decrypt the string
        return $this->encrypter->decryptString(base64_encode(
            json_encode(compact("iv", "value", "mac"), JSON_UNESCAPED_SLASHES)
        ));
    }

    protected function decodeEncryptedPayload(string $payload): object
    {
        $decoded = json_decode(base64_decode($payload));

        if ($decoded === false) {
            throw new InvalidArgumentException('Invalid payload provided');
        }

        return $decoded;
    }

    protected function base64Encode(string $data): string
    {
        $encoded = strtr(base64_encode($data), '+/', '-_');

        return rtrim($encoded, '=');
    }

    protected function base64Decode(string $data): string
    {
        $decoded = base64_decode(strtr($data, '-_', '+/'), true);

        if ($decoded === false) {
            throw new InvalidArgumentException('Invalid data provided');
        }

        return $decoded;
    }

    public function getSecret(): string
    {
        return $this->secret;
    }

    public function getCipher(): ?string
    {
        return $this->cipher;
    }
}
