<?php

declare(strict_types = 1);

namespace JuicyCodes\Signer;

use DateInterval;
use DateTimeInterface;
use Illuminate\Http\Request;
use Illuminate\Routing\Route;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\InteractsWithTime;
use Illuminate\Support\Str;

class Signer
{
    use InteractsWithTime;

    protected Encrypter $encrypter;

    public function __construct(string $secret, string $cipher)
    {
        $this->encrypter = new Encrypter($secret, $cipher);
    }

    public function sign(string $name, array $parameters = [], int|DateInterval|DateTimeInterface $expiration = null, bool $absolute = true): string
    {
        $expiration = $this->availableAt($expiration ?? $this->getDefaultExpiration());
        $signature  = $this->getSignatureHash($name, $parameters, $expiration);

        // Encrypt all the route parameters
        $parameters = array_map(fn($value) => $this->encrypter->encrypt($value), $parameters);

        // Append signature to route parameters
        $parameters["signature"] = $this->encrypter->encrypt(
            sprintf("%s.%s", $expiration, $signature)
        );

        return route($name, $parameters, $absolute);
    }

    /**
     * @psalm-suppress PossiblyInvalidCast
     */
    public function hasValidSignature(Request $request): bool
    {
        try {
            /** @var Route $route */
            $route = $request->route();

            // Decrypt all the parameters
            $this->decryptParameters($route);

            // Get current signature & expiration
            $signature = $route->parameter("signature");
            [$expiration, $signature] = explode(".", (string) $signature);

            // Check if link is expired & is the signature valid
            return $this->signatureHasNotExpired((int) $expiration)
                && hash_equals($this->getRouteSignatureHash($route, (int) $expiration), $signature);
        } catch (\Throwable) {
            return false;
        }
    }

    protected function getSignatureHash(string $name, array $parameters, int $expiration): string
    {
        $payload = json_encode([
            $name,
            $expiration,
            $this->getHeaders(),
            array_values($parameters),
        ]);

        // Create a MAC for the generated payload
        return hash_hmac("md5", $payload, $this->encrypter->getSecret());
    }

    protected function signatureHasNotExpired(int $expires): bool
    {
        return $expires > Carbon::now()->getTimestamp();
    }

    protected function getRouteSignatureHash(Route $route, int $expiration): string
    {
        $parameters = Arr::except($route->parameters(), "signature");

        return $this->getSignatureHash((string) $route->getName(), $parameters, $expiration);
    }

    protected function decryptParameters(Route $route): void
    {
        foreach ($route->parameters() as $name => $value) {
            $route->setParameter($name, $this->encrypter->decrypt($value));
        }
    }

    protected function getHeaders(): array
    {
        $headers = app(Request::class)->headers;
        $random  = static fn(): string => Str::random(10);

        return [
            $headers->get("User-Agent", $random()),
            $headers->get("Accept-Language", $random()),
        ];
    }

    private function getDefaultExpiration(): DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function encrypter(): Encrypter
    {
        return $this->encrypter;
    }
}
