<?php


namespace JuicyCodes\Updater\Support\Traits;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use JuicyCodes\Updater\Backups\Backup;
use JuicyCodes\Updater\Exceptions\AppBackupFailedException;
use JuicyCodes\Updater\Exceptions\RemoveAppBackupFailedException;
use Spatie\Backup\BackupDestination\BackupDestinationFactory;
use Spatie\Backup\Tasks\Backup\DbDumperFactory;
use Spatie\Backup\Tasks\Backup\FileSelection;

trait HasBackupTrait
{
    /**
     * @return string|null
     * @throws AppBackupFailedException
     */
    public function createApplicationBackup(): ?string
    {
        try {
            $backup = new Backup();
            $config = config("backup");

            $backup->setFileSelection(static::createFileSelection($config['backup']['source']['files']))
                ->setDbDumpers(static::createDbDumpers($config['backup']['source']['databases']))
                ->setBackupDestinations(BackupDestinationFactory::createFromArray($config['backup']));

            $filePath = $backup->backup();
        } catch (Exception $e) {
            throw new AppBackupFailedException($e->getMessage());
        }

        return $filePath ?? null;
    }

    protected static function createFileSelection(array $sourceFiles): FileSelection
    {
        return FileSelection::create($sourceFiles['include'])
            ->excludeFilesFrom($sourceFiles['exclude'])
            ->shouldFollowLinks(isset($sourceFiles['follow_links']) && $sourceFiles['follow_links'])
            ->shouldIgnoreUnreadableDirs(Arr::get($sourceFiles, 'ignore_unreadable_directories', false));
    }

    protected static function createDbDumpers(array $dbConnectionNames): Collection
    {
        return collect($dbConnectionNames)->mapWithKeys(function (string $dbConnectionName) {
            return [$dbConnectionName => DbDumperFactory::createFromConnection($dbConnectionName)];
        });
    }

    /**
     * @return void
     * @throws RemoveAppBackupFailedException
     */
    public function removeApplicationBackup(): void
    {
        try {
            $backupDestinations = BackupDestinationFactory::createFromArray(config('backup.backup'));

            foreach ($backupDestinations as $backupDestination) {
                if ($backupDestination->isReachable()) {
                    $backups = $backupDestination->backups();
                    foreach ($backups as $backup) {
                        /** @var Backup $backup */
                        $backup->delete();
                    }
                }
            }
        } catch (Exception $exception) {
            throw new RemoveAppBackupFailedException($exception->getMessage());
        }
    }
}
