<?php

namespace JuicyCodes\Updater;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use JuicyCodes\Updater\Exceptions\FailedAppUpdateException;
use JuicyCodes\Updater\Exceptions\MissingAppDataException;
use JuicyCodes\Updater\Exceptions\MissingDownloadPackageException;
use JuicyCodes\Updater\Exceptions\UnableFetchAvailableUpdates;
use JuicyCodes\Updater\Support\SelfUpdater;
use JuicyCodes\Updater\Support\Traits\HasBackupTrait;
use JuicyCodes\Updater\Support\UpdatePackage;

class AppUpdater
{
    use HasBackupTrait;

    protected string $version;

    protected string $codename;

    protected string $license;

    /**
     * Fetch available updates
     *
     * @return Collection|null
     * @throws GuzzleException
     * @throws MissingAppDataException
     * @throws UnableFetchAvailableUpdates
     */
    public function getAvailableUpdates(): ?Collection
    {
        if ($this->isAppDataMissing()) {
            throw new MissingAppDataException("Missing app data.");
        }

        try {
            $packages = Cache::has("packages") ? Cache::get("packages") : collect();

            if (!(Cache::has("packages"))) {
                $client   = new Client();
                $baseUrl  = "http://juicycodes.public/api";
                $response = $client->get(sprintf("%s/updates/%s/%s/%s", $baseUrl, $this->codename, $this->license, $this->version));
                $response = json_decode((string) $response->getBody(), false, 512, JSON_THROW_ON_ERROR);

                if ($response->success) {
                    foreach ($response->updates as $update) {
                        $updatePackage             = new UpdatePackage();
                        $updatePackage->version    = $update->version;
                        $updatePackage->properties = $update->properties;

                        $packages->push($updatePackage);
                    }
                    Cache::put('packages', $packages, 60 * 60);
                }
            }
        } catch (Exception $exception) {
            throw new UnableFetchAvailableUpdates("Unable fetching available updates");
        }

        return $packages ?? null;
    }

    /**
     * @param string $version
     * @param string $codename
     * @param string $license
     * @return void
     */
    public function setAppData(string $version, string $codename, string $license): void
    {
        $this->version  = $version;
        $this->license  = $license;
        $this->codename = $codename;
    }

    /**
     * @param string $filePath
     * @return bool
     * @throws FailedAppUpdateException
     * @throws MissingDownloadPackageException
     */
    public function update(string $filePath): bool
    {
        if (!File::exists($filePath)) {
            throw new MissingDownloadPackageException("Missing app update package");
        }

        try {
            if ($extract = SelfUpdater::extract($filePath)) {
                $remove  = SelfUpdater::remove();
                $execute = SelfUpdater::callback();
            }

            unlink($filePath);
            $update = $extract && ($remove ?? false) && ($execute ?? false);
            File::deleteDirectory(pathinfo($filePath, PATHINFO_DIRNAME));

            if ((bool) $update) {
                Cache::forget("packages");
            }
        } catch (Exception $exception) {
            throw new FailedAppUpdateException("Unable to update application");
        }

        return (bool) ($update ?? false);
    }

    /**
     * @return bool
     */
    private function isAppDataMissing(): bool
    {
        return empty($this->version) || empty($this->codename) || empty($this->license);
    }

    /**
     * @return string|null
     */
    public function getAppVersion(): ?string
    {
        return $this->version;
    }

    /**
     * @return string|null
     */
    public function getAppLicense(): ?string
    {
        return $this->license;
    }

    /**
     * @return string|null
     */
    public function getAppCodename(): ?string
    {
        return $this->codename;
    }
}
