<?php

declare(strict_types = 1);

namespace JuicyCodes\Plugin\Support\Traits;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use JuicyCodes\Plugin\Contracts\PluginCore;
use JuicyCodes\Plugin\Contracts\PluginMigration;
use JuicyCodes\Plugin\Contracts\PluginNav;
use JuicyCodes\Plugin\Contracts\PluginRoutes;
use JuicyCodes\Plugin\Contracts\PluginScheduler;
use JuicyCodes\Plugin\Support\ClassMapGenerator;
use ReflectionClass;
use Symfony\Component\Finder\Finder;

trait FindPlugins
{
    protected function foundPlugins(): Collection
    {
        return $this->plugins;
    }

    protected function findPlugins(): void
    {
        if (!File::exists($cache = $this->pluginCachePath())) {
            $this->findAndMapPlugins();
        } else {
            $cached = json_decode(File::get($cache), false);

            $this->plugins = collect($cached)->recursive(2);
            $this->findAndMapPluginsIfMissing();
        }
    }

    public function findAndMapPlugins(): void
    {
        $mappedClasses = collect($this->mapPluginClasses());
        $this->plugins = $mappedClasses->transform(function (array $classes, string $plugin) {
            $active = $this->isPluginActive($plugin);

            return compact("active", "classes");
        })->recursive(2);

        $this->cacheMappedPlugins();
    }

    public function findAndMapPluginsIfMissing(): void
    {
        $missingPlugins = $this->plugins->keys()
            ->reject(fn(string $name) => File::exists(app_path("Plugins/{$name}")));

        if ($missingPlugins->isNotEmpty()) {
            $this->findAndMapPlugins();
        }
    }

    protected function mapPluginClasses(): array
    {
        File::ensureDirectoryExists($path = app_path("Plugins"));
        $filePaths = $this->findClassFilePaths($path);
        $classMaps = ClassMapGenerator::createMap($filePaths, null, null, "App\Plugins");

        return collect($classMaps)
            ->mapWithKeys(function (string $path, string $name) {
                $type = $this->getClassType($name);

                return [$path => compact("type", "name")];
            })
            ->filter(fn(array $class) => $class["type"] !== null)
            ->groupBy(fn(array $_class, string $path) => $this->getPluginName($path))
            ->filter(fn(Collection $classes) => $classes->where("type", "core")->count() === 1)
            ->toArray();
    }

    private function getClassType(string $name): ?string
    {
        try {
            if (class_exists($name)) {
                $class = new ReflectionClass($name);

                if ($class->isSubclassOf(Command::class)) {
                    return "command";
                }

                if ($class->isSubclassOf(PluginCore::class)) {
                    return "core";
                }

                if ($class->isSubclassOf(PluginRoutes::class)) {
                    return "route";
                }

                if ($class->isSubclassOf(PluginScheduler::class)) {
                    return "scheduler";
                }

                if ($class->implementsInterface(PluginNav::class)) {
                    return "navigation";
                }

                if ($class->isSubclassOf(PluginMigration::class)) {
                    return "migration";
                }
            }
        } catch (\Exception $e) {
        }

        return null;
    }

    protected function cacheMappedPlugins(): void
    {
        File::put(
            $this->pluginCachePath(),
            $this->plugins->toJson(JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
        );
    }

    protected function pluginCachePath(): string
    {
        return app()->bootstrapPath("cache/plugins.json");
    }

    protected function getPluginName(string $class): string
    {
        $basePath = app_path("Plugins/");

        return Str::before(str_replace($basePath, "", $class), "/");
    }

    protected function findClassFilePaths(string $path): Finder
    {
        return Finder::create()
            ->files()
            ->in($path)
            ->followLinks()
            ->name('/\.(php|inc|hh)$/')
            ->notPath("resources/vendor");
    }
}
