<?php

declare(strict_types=1);

namespace JuicyCodes\Plugin;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class Assets
{
    /**
     * @psalm-suppress PossiblyUndefinedMethod
     */
    public function registerBladeDirective(): void
    {
        app("blade.compiler")->directive("assets", function (string $position) {
            return "<?php echo app('assets')->renderAssets($position); ?>";
        });
    }

    /**
     * @param string       $position
     * @param array|string $assets
     * @param int|null     $offset
     */
    public function register(string $position, $assets, int $offset = null): void
    {
        $assets   = Arr::wrap($assets);
        $existing = $this->getAssets($position);

        $elements = $offset === null
            ? array_merge($existing, $assets)
            : $this->spliceAssets($existing, $offset, $assets);

        config()->set("assets.assets.{$position}", $elements);
    }

    public function renderAssets(string $position): string
    {
        $html   = [];
        $assets = $this->getAssets($position);
        foreach ($assets as $asset) {
            if (Str::endsWith($asset, ".js")) {
                $html[] = $this->buildScriptTag($asset);
            }

            if (Str::endsWith($asset, ".css")) {
                $html[] = $this->buildStyleTag($asset);
            }
        }

        $html[] = null;

        return implode("\n", $html);
    }

    protected function getAssets(string $position): array
    {
        return config()->get("assets.assets.{$position}", []);
    }

    protected function buildStyleTag(string $style): string
    {
        return sprintf(
            '<link href="%s" rel="stylesheet" type="text/css"/>',
            $this->generatePath($style)
        );
    }

    protected function buildScriptTag(string $script): string
    {
        return sprintf(
            '<script src="%s" type="application/javascript"></script>',
            $this->generatePath($script)
        );
    }

    protected function generatePath(string $asset): string
    {
        $callback = $this->shouldUseMix() ? "mix" : "asset";

        if (!$this->shouldUseMix() && $this->shouldUseCacheBuster()) {
            $asset = sprintf("%s?cb=%s", $asset, crc32(microtime()));
        }

        try {
            return $callback($asset);
        } catch (\Exception $exception) {
            return $asset;
        }
    }

    protected function spliceAssets(array $existing, int $offset, array $assets): array
    {
        array_splice($existing, $offset, 0, $assets);

        return $existing;
    }

    protected function shouldUseMix(): bool
    {
        return config("assets.config.use_mix", false);
    }

    protected function shouldUseCacheBuster(): bool
    {
        return config("assets.config.cache_buster", false);
    }
}
