<?php

declare(strict_types = 1);

namespace JuicyCodes\Plugin\Support\Traits;

use Illuminate\Console\Application;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Container\Container;
use Illuminate\Contracts\Foundation\CachesRoutes;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;
use JuicyCodes\Plugin\Contracts\PluginCore;
use JuicyCodes\Plugin\Contracts\PluginRoutes;
use JuicyCodes\Plugin\Contracts\PluginScheduler;

trait BootPlugins
{
    protected bool $booted = false;

    protected array $coreInstances = [];

    protected function bootPlugins(): void
    {
        if ($this->isBooted()) {
            return;
        }

        $this->setBooted(true);
        $plugins = $this->foundPlugins()
            ->filter(fn(Collection $_plugin, string $plugin) => $this->isPluginActive($plugin));

        $this->registerAndRefreshRoutes($plugins);
        $plugins->each(function (Collection $plugin, string $name): void {
            $classes  = $plugin->get("classes");
            $instance = $this->getPluginCore($classes, $name);
            $instance->registerPlugin($name);

            if (!$instance->license()->isValid()) {
                $this->deactivatePlugin($name, true);
            } else {
                $this->loadTranslations($name);
                $this->loadPluginBladeViews($name);

                // Boot plugin
                $instance->boot();

                $this->registerPluginCommands($classes);
                $this->registerPluginSchedulers($classes);
            }
        });
    }

    /**
     * @param string $plugin
     * @psalm-suppress PossiblyUndefinedMethod
     */
    protected function loadTranslations(string $plugin): void
    {
        $directory = app_path("Plugins/{$plugin}/resources/lang");

        if (File::exists($directory)) {
            app("translator")->addNamespace(Str::snake($plugin), $directory);
        }
    }

    /**
     * @param string $plugin
     * @psalm-suppress PossiblyUndefinedMethod
     */
    public function loadPluginBladeViews(string $plugin): void
    {
        $directory = app_path("Plugins/{$plugin}/resources/views");

        if (File::exists($directory)) {
            app("view")->addNamespace(Str::snake($plugin), $directory);
        }
    }

    /**
     * @param Collection $plugins
     * @psalm-suppress PossiblyUndefinedMethod
     */
    public function registerAndRefreshRoutes(Collection $plugins): void
    {
        $app = Container::getInstance();
        if (!($app instanceof CachesRoutes && $app->routesAreCached())) {
            $plugins->each(function (Collection $plugin, string $name): void {
                $name    = Str::snake($name);
                $classes = $plugin->get("classes");
                self::registerRoutes($name, $classes);
            });

            app('router')->getRoutes()->refreshNameLookups();
            app('router')->getRoutes()->refreshActionLookups();
        }
    }

    protected function registerRoutes(string $plugin, Collection $classes): void
    {
        $classes = $classes->where("type", "route")->pluck("name");
        $classes->each(function (string $class) use ($plugin): void {
            /** @var PluginRoutes $instance */
            $instance = $this->instance($class);
            Route::name("{$plugin}::")->group(fn() => $instance->map());
        });
    }

    protected function registerPluginCommands(Collection $classes): void
    {
        $commands = $classes->where("type", "command")->pluck("name");
        Application::starting(function (Application $artisan) use ($commands): void {
            $artisan->resolveCommands($commands->toArray());
        });
    }

    protected function registerPluginSchedulers(Collection $classes): void
    {
        app()->afterResolving(Schedule::class, function (Schedule $schedule) use ($classes): void {
            $classes
                ->where("type", "scheduler")->pluck("name")
                ->each(function (string $class) use ($schedule): void {
                    /** @var PluginScheduler $instance */
                    $instance = new $class();
                    $instance->schedule($schedule);
                });
        });
    }

    protected function getPluginCore(Collection $classes = null, string $plugin = null): PluginCore
    {
        $classes ??= $this->foundPlugins()->get($plugin)->get("classes");

        return $this->instance($classes->where("type", "core")->pluck("name")->first());
    }

    public function isBooted(): bool
    {
        return $this->booted;
    }

    public function setBooted(bool $booted): void
    {
        $this->booted = $booted;
    }
}
