<?php

declare(strict_types=1);

namespace JuicyCodes\Plugin\Support;

use Illuminate\Database\Connection;
use Illuminate\Database\Schema\Grammars\Grammar;
use Illuminate\Database\Seeder;
use JuicyCodes\Plugin\Contracts\PluginMigration;

class Migrator
{
    /**
     * Run a migration inside a transaction if the database supports it.
     *
     * @param PluginMigration $migration
     * @param string          $method
     * @return void
     * @throws \Throwable
     */
    public static function runMigration(PluginMigration $migration, string $method): void
    {
        /** @psalm-suppress PossiblyUndefinedMethod */
        $connection = app("db")->connection(
            $migration->getConnection()
        );

        $callback = static function () use ($migration, $method): void {
            if ($migration->shouldMigrate($method)) {
                $migration->{$method}();
                self::runSeeders($migration->seeders());
            }
        };

        self::getSchemaGrammar($connection)->supportsSchemaTransactions()
        && $migration->withinTransaction
            ? $connection->transaction($callback)
            : $callback();
    }

    /**
     * Get a seeder instance from the container.
     *
     * @param array $seeders
     * @return void
     */
    protected static function runSeeders(array $seeders): void
    {
        foreach ($seeders as $seeder) {
            /** @var Seeder $instance */
            $instance = new $seeder();
            $instance->__invoke();
        }
    }

    /**
     * Get the schema grammar out of a migration connection.
     *
     * @param Connection $connection
     * @return Grammar
     */
    protected static function getSchemaGrammar(Connection $connection): Grammar
    {
        /** @psalm-suppress DocblockTypeContradiction */
        if (is_null($grammar = $connection->getSchemaGrammar())) {
            $connection->useDefaultSchemaGrammar();

            $grammar = $connection->getSchemaGrammar();
        }

        return $grammar;
    }
}
