<?php

declare(strict_types = 1);

namespace JuicyCodes\Plugin\Support\Traits;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use JuicyCodes\Plugin\Contracts\PluginCore;
use JuicyCodes\Plugin\Contracts\PluginMigration;
use JuicyCodes\Plugin\Exceptions\PluginNotFoundException;

trait ManagePlugins
{
    public function getPlugins(): Collection
    {
        return $this->foundPlugins()
            ->map(function (Collection $_plugin, string $identifier): PluginCore {
                $instance = $this->getPluginCore(plugin: $identifier);
                $instance->registerPlugin($identifier);

                return $instance;
            })
            ->sortBy(fn(PluginCore $plugin) => $plugin->getName());
    }

    /**
     * @throws PluginNotFoundException
     */
    public function activatePlugin(string $plugin): void
    {
        if (!$this->foundPlugins()->has($plugin)) {
            PluginNotFoundException::throw($plugin);
        } else {
            $instance = $this->getPluginCore(plugin: $plugin);
            $instance->verifyLicense();

            $this->plugins[$plugin]["active"] = true;

            $this->cacheMappedPlugins();
            $this->publishCompiledAssets($plugin);
            $this->runMigrations($plugin, PluginMigration::UP);

            $instance->setActive(true);
            $instance->onActivate();
        }
    }

    /**
     * @throws PluginNotFoundException
     */
    public function deactivatePlugin(string $plugin, bool $simple = false): void
    {
        if (!$this->foundPlugins()->has($plugin)) {
            PluginNotFoundException::throw($plugin);
        } else {
            $this->plugins[$plugin]["active"] = false;
            $this->cacheMappedPlugins();

            $instance = $this->getPluginCore(plugin: $plugin);
            $instance->setActive(false);

            if ($simple === false) {
                $this->removePublishedAssets($plugin);
                $this->runMigrations($plugin, PluginMigration::DOWN);
                $instance->onDeactivate();
            }
        }
    }

    public function isPluginActive(string $plugin): bool
    {
        return $this->plugins[$plugin]["active"] ?? false;
    }

    private function publishCompiledAssets(string $plugin): void
    {
        File::copyDirectory(
            app_path("Plugins/{$plugin}/resources/compiled"),
            public_path("assets/plugins/" . Str::snake($plugin))
        );
    }

    private function removePublishedAssets(string $plugin): void
    {
        File::deleteDirectory(public_path("assets/plugins/" . Str::snake($plugin)));
    }
}
