<?php

declare(strict_types=1);

namespace JuicyCodes\Plugin;

use Illuminate\Support\Collection;
use JuicyCodes\Plugin\Support\Traits\BootPlugins;
use JuicyCodes\Plugin\Support\Traits\FindPlugins;
use JuicyCodes\Plugin\Support\Traits\ManagePlugins;
use JuicyCodes\Plugin\Support\Traits\RegisterNavigations;
use JuicyCodes\Plugin\Support\Traits\RunMigrations;
use Symfony\Component\Routing\Exception\RouteNotFoundException;

class Plugin
{
    use FindPlugins;
    use BootPlugins;
    use RunMigrations;
    use ManagePlugins;
    use RegisterNavigations;

    protected array $actions;

    protected array $instances;

    protected Collection $plugins;

    /**
     * Plugin constructor.
     */
    public function __construct()
    {
        $this->actions   = [];
        $this->instances = [];
        $this->plugins   = collect([]);
    }

    public function boot(): void
    {
        $this->findPlugins();
        $this->bootPlugins();
    }

    /**
     * Add the given callback to given action
     *
     * @param string   $action
     * @param callable $callback
     */
    public function addAction(string $action, callable $callback): void
    {
        $this->actions[$action][] = $callback;
    }

    /**
     * Execute functions hooked on the given action
     *
     * @param string $action
     * @param mixed  ...$parameters
     */
    public function executeAction(string $action, &...$parameters): void
    {
        if (isset($this->actions[$action])) {
            foreach ($this->actions[$action] as $callback) {
                $callback(...$parameters);
            }
        }
    }

    public function route(string $name): string
    {
        try {
            return route($name);
        } catch (RouteNotFoundException) {
            return "#!route-not-found";
        }
    }

    protected function instance(string $class): mixed
    {
        return $this->instances[$class] ??= new $class();
    }
}
