<?php

namespace JuicyCodes\Generator\Menu;

use Closure;
use Illuminate\Support\Str;

/**
 * Class Item
 *
 * @package JuicyCodes\Generator\Menu
 */
class Item
{
    /**
     * @var string
     */
    protected string $title;

    /**
     * @var string
     */
    protected string $link;

    /**
     * @var string
     */
    protected string $icon;

    /**
     * @var bool
     */
    protected bool $isSeparator;

    /**
     * @var array
     */
    protected array $classes;

    /**
     * @var Builder
     */
    protected Builder $subMenu;

    /**
     * @var string
     */
    protected string $identifier;

    /**
     * @var bool
     */
    protected bool $isActive = false;

    /**
     * Create new Item instance with title & class names
     *
     * @param string $title
     * @param array  $classes
     */
    public function __construct(string $title, array $classes)
    {
        $this->title($title);
        $this->classes = $classes;
        $this->identifier(Str::camel($title));
    }

    /**
     * Attach a submenu to this item
     *
     * @param Closure $callback
     * @return $this
     */
    public function addSubMenu(Closure $callback)
    {
        # Call closure function with menu instance
        $callback($this->subMenu());

        return $this;
    }

    /**
     * Get submenu of this menu item
     *
     * @return Builder
     */
    public function subMenu(): Builder
    {
        if (empty($this->subMenu)) {
            # Get new clean menu instance if it doesn't already exists
            $this->subMenu = menu()->emptyInstance();
        }

        return $this->subMenu;
    }

    /**
     * Get HTML classes for this menu item
     *
     * @return string
     */
    public function classes()
    {
        # If this item is a separator return separator classes
        if ($this->isSeparator()) {
            return $this->classes["separator"];
        }

        $classes = [];
        array_push($classes, $this->classes["default"]);

        # If menu item is active, add `active` class
        if ($this->isActive()) {
            $classes[] = $this->classes["active"];
        }

        # IF menu item has a submenu, add `submenu` classes
        if (!empty($this->subMenu)) {
            $classes[] = $this->classes["submenu"];
        }

        return implode(" ", $classes);
    }

    /**
     * Check whether this item is active
     *
     * @return bool
     */
    public function isActive(): bool
    {
        if ($this->isActive === true) {
            return $this->isActive;
        }

        # If item has no submenu then compare item link & current page link
        if (!$this->hasSubMenu()) {
            return ($this->link() === url()->current() || Str::startsWith(url()->current(), $this->link()));
        }

        # Check whether current tem has any active children
        return !empty($this->getActiveChild());
    }

    /**
     * Check whether this menu item has a submenu
     *
     * @return bool
     */
    public function hasSubMenu(): bool
    {
        return !empty($this->subMenu);
    }

    /**
     * Check if this Item has any active children
     *
     * @return array
     */
    public function getActiveChild(): array
    {
        return array_filter($this->subMenu()->getItems(), function (self $item) {
            return $item->isActive();
        });
    }

    /**
     * Set or Get item title
     *
     * @param string|null $title
     * @return $this|string
     */
    public function title(?string $title = null)
    {
        # Return value if none provided
        if (is_null($title)) {
            return $this->title;
        }

        return $this->setAttribute("title", $title);
    }

    /**
     * Set or Get item link
     *
     * @param string|null $link
     * @return $this|string
     */
    public function link(?string $link = null)
    {
        # Return value if none provided
        if (is_null($link)) {
            return $this->link;
        }

        return $this->setAttribute("link", $link);
    }

    /**
     * Set or Get item icon
     *
     * @param string|null $icon
     * @return $this|Icon
     */
    public function icon(?string $icon = null)
    {
        # Return value if none provided
        if (is_null($icon)) {
            return new Icon($this->icon);
        }

        return $this->setAttribute("icon", $icon);
    }

    /**
     * Set or Get item identifier
     *
     * @param string|null $identifier
     * @return $this|string
     */
    public function identifier(?string $identifier = null)
    {
        # Return value if none provided
        if (is_null($identifier)) {
            return $this->identifier;
        }

        return $this->setAttribute("identifier", $identifier);
    }

    /**
     * Set or Get Item's separator status
     *
     * @param bool|null $isSeparator
     * @return $this|bool
     */
    public function isSeparator(?bool $isSeparator = null)
    {
        # Return value if none provided
        if (!is_bool($isSeparator)) {
            return $this->isSeparator;
        }

        return $this->setAttribute("isSeparator", $isSeparator);
    }

    /**
     * Set the menu item as "active"
     *
     * @return void
     */
    public function setAsActive(): void
    {
        $this->isActive = true;
    }

    /**
     * Set item property value
     *
     * @param string      $name
     * @param string|bool $value
     * @return $this
     */
    protected function setAttribute(string $name, $value): self
    {
        $this->{$name} = $value;

        return $this;
    }
}
