<?php

namespace JuicyCodes\Generator\Breadcrumbs;

use Closure;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use JuicyCodes\Generator\Breadcrumbs\Exceptions\DuplicateBreadcrumbException;

/**
 * Class Breadcrumbs
 *
 * @date 12-08-2020
 * @package JuicyCodes\Generator\Breadcrumbs
 */
class Breadcrumbs
{
    /**
     * @var Collection
     */
    private Collection $trails;

    /**
     * @var string
     */
    private string $template;

    /**
     * Breadcrumbs constructor.
     */
    public function __construct()
    {
        $this->trails = new Collection();
    }

    /**
     * Register singleton & blade directive
     */
    public static function register(): void
    {
        app()->singleton("breadcrumbs", Breadcrumbs::class);
        app("blade.compiler")->directive("breadcrumbs", function ($expression) {
            return "<?php echo e(breadcrumbs()->render($expression)); ?>";
        });
    }

    /**
     * @param string      $label
     * @param Closure     $callback
     * @param string|null $identifier
     * @param string|null $parent
     * @throws \Throwable
     */
    public function for(string $label, Closure $callback, string $identifier = null, string $parent = null): void
    {
        $identifier = $identifier ?? Str::snake(str_replace("&", "", $label));

        // Prevent duplicate breadcrumbs
        throw_if($this->trails->contains($identifier), new DuplicateBreadcrumbException(
            "A breadcrumb trail with the identifier [{$identifier}] already exists."
        ));

        $this->trails->put($identifier, new Trail(
            $label,
            $callback,
            $identifier,
            $this,
            $parent
        ));
    }

    /**
     * @param string $identifier
     * @param mixed  ...$params
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function render(string $identifier, ...$params)
    {
        /** @var Trail $breadcrumb */
        $breadcrumb = $this->trails->get($identifier);

        $breadcrumb->setParams($params);
        $breadcrumbs = collect([$breadcrumb]);

        while ($breadcrumb->hasParent()) {
            $breadcrumb = $this->trails->get($breadcrumb->getParent());
            $breadcrumbs->prepend($breadcrumb);
        }

        return view($this->template, compact("breadcrumbs"));
    }

    /**
     * @param string $template
     */
    public function setTemplate(string $template): void
    {
        $this->template = $template;
    }

    /**
     * @param string $identifier
     * @return mixed
     * @throws \ErrorException
     */
    public function __get(string $identifier)
    {
        /** @var Trail $trail */
        $trail = $this->trails->first(function (Trail $trail) use ($identifier) {
            return $trail->getIdentifier() === $identifier;
        });

        if (!empty($trail)) {
            return $trail;
        }

        # Throw an exception if no breadcrumb trail is found
        throw  new \ErrorException(sprintf('Undefined property: %s::$%s', __CLASS__, $identifier));
    }
}
