<?php

namespace JuicyCodes\Generator\Menu;

use Illuminate\Support\Collection;

/**
 * Class Builder
 *
 * @package JuicyCodes\Generator\Menu
 */
class Builder
{
    /**
     * @var array
     */
    protected array $classes;

    /**
     * @var Collection
     */
    protected Collection $items;

    /**
     * Create new Builder instance with default class name
     */
    public function __construct()
    {
        $this->items   = collect([]);
        $this->classes = [
            "active"    => "menu__item--active",
            "default"   => "menu__item",
            "submenu"   => "menu__item--submenu",
            "separator" => "menu-separator kt-separator--space-sm",
        ];
    }

    /**
     * Add new menu item with or without route name
     *
     * @param string      $title
     * @param string|null $route
     * @return Item
     */
    public function add(string $title, ?string $route = null): Item
    {
        # Create new Item with title & class names
        $item = new Item($title, $this->classes);
        $item->isSeparator(false);

        # Set item link via route name if given
        if (isset($route)) {
            $item->link(route($route));
        }

        $this->items->push($item);

        return $item;
    }

    /**
     * Add a Separator Item
     *
     * @param string $title
     * @return Item
     */
    public function addSeparator(string $title): Item
    {
        # Create new Item with title & class names
        $item = new Item($title, $this->classes);
        $item->isSeparator(true);

        $this->items->push($item);

        return $item;
    }

    /**
     * Return menu items
     *
     * @return array|Item[]
     */
    public function getItems(): array
    {
        return $this->items->toArray();
    }

    /**
     * Return the currently active menu Item
     *
     * @return Item|null
     */
    public function activeItem(): ?Item
    {
        foreach ($this->getItems() as $item) {
            # If current item is not active then ignore it
            if ($item->isActive() === false) {
                continue;
            }

            # If the active item has submenu then return the active child else return the Item
            $active = !$item->hasSubMenu() ? $item : head($item->getActiveChild());

            break;
        }

        return $active ?? null;
    }

    /**
     * Set menu class names
     *
     * @param array $classes
     */
    public function setClasses(array $classes): void
    {
        $this->classes = array_merge($this->classes, $classes);
    }

    /**
     * Return empty instance with class names
     */
    public function emptyInstance(): self
    {
        $menu = new self();
        $menu->setClasses($this->classes);

        return $menu;
    }

    /**
     * @param string $name
     * @return mixed
     * @throws \ErrorException
     */
    public function __get(string $name)
    {
        /** @var Item|null $item */
        $item = $this->items->first(function (Item $item) use ($name) {
            return $item->identifier() === $name;
        });

        # Get submenu of this menu item
        if ($item !== null) {
            return $item->hasSubMenu() ? $item->subMenu() : $item;
        }

        # Throw an exception if no menu item is found
        throw  new \ErrorException(sprintf('Undefined property: %s::$%s', __CLASS__, $name));
    }
}
