<?php

namespace JuicyCodes\Logger;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use JuicyCodes\Logger\Models\Log;

class Logger
{
    public const INFO = "info";
    public const ERROR = "error";
    public const SUCCESS = "success";

    protected Log $model;

    protected ?Model $writer;

    public function __construct(?Model $writer = null)
    {
        $this->writer = $writer;
        $this->createNewModelInstance();
    }

    public function writtenFor(?Model $model): self
    {
        $this->model()->subject()->associate($model);

        return $this;
    }

    public function writtenBy(?Model $model): self
    {
        $this->model()->writer()->associate($model);

        return $this;
    }

    public function loggedAt(Carbon $loggedAt): self
    {
        $this->model()->created_at = $loggedAt;

        return $this;
    }

    public function withType(string $type): self
    {
        $this->model()->type = $type;

        return $this;
    }

    public function withLevel(string $level): self
    {
        $this->model()->level = $level;

        return $this;
    }

    public function withContext(array $context): self
    {
        $this->model()->context = $context;

        return $this;
    }

    public function write(string $message): Log
    {
        $model = $this->model();
        $model->setAttribute("message", $message);
        $model->save();

        $this->createNewModelInstance();

        return $model;
    }

    public function info(string $message, array $context = []): Log
    {
        return $this
            ->withLevel(self::INFO)
            ->withContext($context)
            ->write($message);
    }

    public function error(string $message, array $context = []): Log
    {
        return $this->withLevel(self::ERROR)
            ->withContext($context)
            ->write($message);
    }

    public function success(string $message, array $context = []): Log
    {
        return $this->withLevel(self::SUCCESS)
            ->withContext($context)
            ->write($message);
    }

    public function model(): Log
    {
        return $this->model;
    }

    protected function createNewModelInstance(): void
    {
        $this->model = new Log();
        $this->withType("default");
        $this->withLevel(self::INFO);
        $this->writtenBy($this->writer);
    }
}
