<?php

namespace JuicyCodes\KV\Repositories;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

abstract class KeyValueRepository
{
    protected ?Collection $records = null;

    protected ?array $dottedRecords = null;

    /**
     * @return Builder|HasMany
     */
    abstract public function getQuery();

    abstract protected function getCacheKey(): string;

    /**
     * @param string $key
     * @param mixed  $default
     * @return mixed
     */
    public function get(string $key, $default = null)
    {
        $this->dottedRecords ??= Arr::dot($this->records());

        return $this->dottedRecords[$key] ?? value($default);
    }

    /**
     * @param string|array $key
     * @param mixed        $value
     */
    public function set($key, $value = null): void
    {
        $keys = is_array($key) ? $key : [$key => $value];

        foreach ($keys as $key => $value) {
            $this->getQuery()->updateOrCreate(
                compact("key"),
                compact("value"),
            );
        }

        // Remove cached settings
        $this->cleanCachedRecords();
    }

    /**
     * Remove/delete the specified setting value.
     *
     * @param string $key
     * @return bool
     */
    public function remove(string $key): bool
    {
        $isRemoved = (bool) $this->getQuery()->where('key', $key)->delete();

        if ($isRemoved === true) {
            $this->cleanCachedRecords();
        }

        return $isRemoved;
    }

    /**
     * @return Collection
     */
    public function records(): Collection
    {
        return $this->records ??= Cache::rememberForever(
            $this->getCacheKey(),
            fn(): Collection => $this->getQuery()->pluck("value", "key")
        );
    }

    protected function cleanCachedRecords(): void
    {
        Cache::forget($this->getCacheKey());
        unset($this->records, $this->dottedRecords);
    }
}
