<?php

namespace JuicyCodes\KV\Repositories;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Traits\ForwardsCalls;
use JuicyCodes\KV\Models\Meta;

class MetaRepository extends KeyValueRepository
{
    use ForwardsCalls;

    protected Model $model;

    protected string $metaTable;

    /**
     * MetaRepository constructor.
     *
     * @param Model  $model
     * @param string $metaTable
     */
    public function __construct(Model $model, string $metaTable)
    {
        $this->setModel($model);
        $this->setMetaTable($metaTable);

        if (empty($model->uuid)) {
            throw new \RuntimeException("UUID attribute is required when using metable trait.");
        }
    }

    public function getQuery(): HasMany
    {
        $metaModel = new Meta();
        $metaModel->setTable($this->getMetaTable());

        return (new HasMany(
            $metaModel->newQuery(),
            $this->getModel(),
            $this->getModel()->getForeignKey(),
            $this->getModel()->getKeyName()
        ));
    }

    protected function getCacheKey(): string
    {
        return "kv::meta_{$this->model->uuid}";
    }

    /**
     * @return Model
     */
    public function getModel(): Model
    {
        return $this->model;
    }

    /**
     * @param Model $model
     */
    public function setModel(Model $model): void
    {
        $this->model = $model;
    }

    /**
     * @return string
     */
    public function getMetaTable(): string
    {
        return $this->metaTable;
    }

    /**
     * @param string $metaTable
     */
    public function setMetaTable(string $metaTable): void
    {
        $this->metaTable = $metaTable;
    }

    /**
     * Handle dynamic method calls into the model.
     *
     * @param string $method
     * @param array  $parameters
     * @return mixed
     */
    public function __call(string $method, array $parameters)
    {
        return $this->forwardCallTo($this->getQuery()->newQuery(), $method, $parameters);
    }
}
