<?php

namespace JuicyCodes\KV\Repositories;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\QueryException;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use JuicyCodes\KV\ValueSetException;

abstract class KeyValueRepository
{
    protected ?Collection $records = null;

    protected ?array $dottedRecords = null;

    /**
     * @return Builder|HasMany
     */
    abstract public function getQuery();

    abstract protected function getCacheKey(): string;

    /**
     * @param string $key
     * @param mixed  $default
     * @return mixed
     */
    public function get(string $key, $default = null)
    {
        $this->dottedRecords ??= Arr::dot($this->records());

        // Reset dotted records to null when empty
        if (empty($this->dottedRecords)) {
            $this->dottedRecords = null;
        }

        return $this->transformValue(
            $this->dottedRecords[$key]
            ?? $this->records()->get($key, $default)
        );
    }

    /**
     * @param string|array $key
     * @param mixed        $value
     * @return bool
     * @throws ValueSetException
     */
    public function set($key, $value = null): bool
    {
        $keys = is_array($key) ? $key : [$key => $value];

        foreach ($keys as $key => $value) {
            try {
                $this->getQuery()->updateOrCreate(
                    compact("key"),
                    compact("value"),
                );
            } catch (QueryException $exception) {
                throw new ValueSetException("Failed to set value of '{$key}'.");
            }
        }

        // Remove cached settings
        $this->cleanCachedRecords();
        return true;
    }

    /**
     * Remove/delete the specified setting value.
     *
     * @param string $key
     * @return bool
     */
    public function remove(string $key): bool
    {
        $isRemoved = (bool) $this->getQuery()->where('key', $key)->delete();

        if ($isRemoved === true) {
            $this->cleanCachedRecords();
        }

        return $isRemoved;
    }

    /**
     * @return Collection
     */
    public function records(): Collection
    {
        try {
            return $this->records ??= Cache::rememberForever(
                $this->getCacheKey(),
                fn(): Collection => $this->getQuery()->pluck("value", "key")
            );
        } catch (\Throwable $exception) {
            return Collection::empty();
        }
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    protected function transformValue($value)
    {
        return !is_array($value)
            ? $value
            : json_decode(json_encode($value));
    }

    protected function cleanCachedRecords(): void
    {
        Cache::forget($this->getCacheKey());
        unset($this->records, $this->dottedRecords);
    }
}
