<?php

namespace JuicyCodes\GeoIp;

use Illuminate\Cache\CacheManager;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use JuicyCodes\GeoIp\Exceptions\InvalidProviderException;
use JuicyCodes\GeoIp\Exceptions\NoPrimaryProviderSpecifiedException;
use JuicyCodes\GeoIp\Providers\AbstractProvider;
use JuicyCodes\GeoIp\Support\GeoData;
use JuicyCodes\GeoIp\Support\IPAddress;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;

class GeoIp
{
    use IPAddress;

    public const PRIMARY_PROVIDER = "primary";

    public const FALLBACK_PROVIDER = "fallback";

    protected Logger $log;

    protected array $config;

    /**
     * GeoIp constructor.
     */
    public function __construct()
    {
        $this->config = Config::get("geoip", []);

        // Create logger instance
        $this->log = new Logger("geoip");
        $this->log->pushHandler(new RotatingFileHandler(
            storage_path('logs/geoip.log'),
            7,
            Logger::ERROR
        ));
    }

    /**
     * @param string|null $ip
     * @return GeoData
     * @throws Exceptions\GeoIpException
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function locate(string $ip = null): GeoData
    {
        $ip ??= $this->ipAddress();

        $cacheKey = "geoip::{$ip}";
        if ($this->cache()->has($cacheKey)) {
            /** @var GeoData $geoData */
            $geoData = $this->cache()->get($cacheKey);
            $geoData->setIsCached(true);
        } else {
            $geoData = $this->getGeoData($ip);
            if ($this->shouldCache($geoData)) {
                Cache::put($cacheKey, $geoData, now()->addDay());
            }
        }

        return $geoData;
    }

    /**
     * @param string $ip
     * @param string $type
     * @return GeoData
     * @throws Exceptions\GeoIpException
     */
    protected function getGeoData(string $ip, $type = self::PRIMARY_PROVIDER): GeoData
    {
        try {
            $provider = $this->getProvider($type);

            return $provider->locate($ip);
        } catch (InvalidProviderException $exception) {
            throw $exception;
        } catch (NoPrimaryProviderSpecifiedException $exception) {
            throw $exception;
        } catch (\Throwable $exception) {
            $this->log->error($exception->getMessage(), [
                "line" => $exception->getLine(),
                "file" => $exception->getFile(),
            ]);
        }

        if ($this->isFallbackSpecified() && $type === self::PRIMARY_PROVIDER) {
            return $this->getGeoData($ip, self::FALLBACK_PROVIDER);
        }

        return new GeoData($ip, null, self::class);
    }

    /**
     * @param string $type
     * @return AbstractProvider
     * @throws InvalidProviderException
     * @throws NoPrimaryProviderSpecifiedException
     */
    private function getProvider(string $type = self::PRIMARY_PROVIDER): AbstractProvider
    {
        $name = $this->config("provider.{$type}");
        if (is_null($name)) {
            throw new NoPrimaryProviderSpecifiedException("No {$type} Geo data provider has been specified.");
        }

        $provider = $this->config("providers.{$name}");
        if ($this->isInvalidProvider($provider)) {
            throw new InvalidProviderException("The specified Geo data provider ({$name}) is not valid.");
        }

        $class = $provider["class"];

        return new $class(array_merge($this->config, $provider));
    }

    /**
     * @return CacheManager
     */
    protected function cache(): CacheManager
    {
        return app("cache");
    }

    /**
     * Get configuration value.
     *
     * @param string $key
     * @param mixed  $default
     * @return mixed
     */
    protected function config(string $key, $default = null)
    {
        return Arr::get($this->config, $key, $default);
    }

    /**
     * @param GeoData $geoData
     * @return bool
     */
    protected function shouldCache(GeoData $geoData): bool
    {
        return $this->config("cache")
            && $geoData->getProviderClass() !== self::class;
    }

    /**
     * @return bool
     */
    protected function isFallbackSpecified(): bool
    {
        return $this->config("provider.fallback") !== null;
    }

    /**
     * @param array|null $provider
     * @return bool
     */
    protected function isInvalidProvider(?array $provider): bool
    {
        return is_null($provider) || empty($provider["class"]);
    }
}
