<?php

namespace JuicyCodes\GeoIp\Support;

use JuicyCodes\GeoIp\Exceptions\GeoIpException;
use JuicyCodes\GeoIp\GeoIp;
use League\ISO3166\ISO3166;

class GeoData
{
    private string $ipAddress;

    private ?string $countryCode;

    private string $providerClass;

    private bool $isCached = false;

    /**
     * GeoData constructor.
     *
     * @param string      $ip
     * @param string|null $countryCode
     * @param string      $providerClass
     * @throws GeoIpException
     */
    public function __construct(string $ip, ?string $countryCode, string $providerClass)
    {
        $this->ipAddress     = $ip;
        $this->countryCode   = $countryCode;
        $this->providerClass = $providerClass;

        if (empty($this->countryCode) && $providerClass !== GeoIp::class) {
            throw new GeoIpException("Failed to fetch geolocation data for '{$ip}' using '{$providerClass}'");
        }
    }

    /**
     * @return string
     */
    public function getIpAddress(): string
    {
        return $this->ipAddress;
    }

    /**
     * @return string|null
     */
    public function getCountryCode(): ?string
    {
        return $this->countryCode;
    }

    /**
     * @return string|null
     */
    public function getCountryName(): ?string
    {
        try {
            $code = $this->getCountryCode();
            $data = (new ISO3166)->alpha2($code);

            return $data["name"] ?? null;
        } catch (\Throwable $e) {
            return null;
        }
    }

    /**
     * @return string
     */
    public function getProviderClass(): string
    {
        return $this->providerClass;
    }

    /**
     * @return bool
     */
    public function isCached(): bool
    {
        return $this->isCached;
    }

    /**
     * @param bool $isCached
     */
    public function setIsCached(bool $isCached): void
    {
        $this->isCached = $isCached;
    }
}
