<?php

namespace JuicyCodes\GeoIp\Support;

trait IpAddressTrait
{
    /**
     * Get client IP address
     *
     * @return string
     */
    public function ipAddress(): string
    {
        # Iterate over the keys until a valid ip is found
        foreach ($this->envKeys() as $key) {
            if ($address = $this->getEnv($key)) {
                foreach ($address as $ip) {
                    if ($this->isValidAddress($ip)) {
                        return $ip;
                    }
                }
            }
        }

        # Return local ip in case no ip found
        return '127.0.0.1';
    }

    /**
     * Get all the IPs associated with the client
     *
     * @return array
     */
    public function ipAddresses()
    {
        $addresses = [];

        # Iterate over the IP keys
        foreach ($this->envKeys() as $key) {
            if ($address = $this->getEnv($key)) {
                array_push($addresses, $address);
            }
        }

        # Keep only unique IPs
        return array_values(array_unique($addresses));
    }

    /**
     * Check if the provided ip is valid & not in private or reserved range
     *
     * @param  $ip
     * @return bool
     */
    protected function isValidAddress($ip)
    {
        return (bool) filter_var(
            $ip,
            FILTER_VALIDATE_IP,
            FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE
        );
    }

    /**
     * Gets the value of an environment variable
     *
     * @param  $key
     * @return array|bool
     */
    protected function getEnv($key)
    {
        # Check if value exists
        if (!empty($address = getenv($key))) {
            # Get environment value by key
            $values = explode(",", $address);
            # Trim all the array item
            return array_map("trim", $values);
        }

        return false;
    }

    /**
     * Get env key that provided client's IP address
     *
     * @return array
     */
    protected function envKeys(): array
    {
        return [
            "HTTP_CF_CONNECTING_IP",
            "HTTP_CLIENT_IP",
            "HTTP_X_FORWARDED_FOR",
            "HTTP_X_FORWARDED",
            "HTTP_FORWARDED_FOR",
            "HTTP_FORWARDED",
            "REMOTE_ADDR",
            "HTTP_X_CLUSTER_CLIENT_IP",
        ];
    }
}
