<?php

namespace JuicyCodes\GeoIp\Providers;

use GeoIp2\Database\Reader;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\File;
use JuicyCodes\GeoIp\Exceptions\GeoIpException;
use JuicyCodes\GeoIp\Support\GeoData;
use PharData;
use Symfony\Component\Finder\Finder;

class MaxMindProvider extends AbstractProvider
{
    /**
     * @param string $ip
     * @return GeoData
     * @throws GeoIpException
     * @throws \GeoIp2\Exception\AddressNotFoundException
     * @throws \MaxMind\Db\Reader\InvalidDatabaseException
     */
    public function locate(string $ip): GeoData
    {
        $this->ensureDatabaseExists();
        $path = $this->getDatabasePath();

        $reader = new Reader($path);
        $record = $reader->country($ip);

        return $this->hydrate($ip, $record->country->isoCode);
    }

    /**
     * @throws GeoIpException
     */
    private function ensureDatabaseExists(): void
    {
        if ($this->shouldDownloadDatabase()) {
            $this->performDatabaseDownload();
            $this->cleanUpDatabaseResources();
        }

        if (!File::exists($this->getDatabasePath())) {
            throw new GeoIpException("No maxmind database exists in local storage.");
        }
    }

    /**
     * @throws GeoIpException
     */
    private function performDatabaseDownload(): void
    {
        try {
            // Make sure that resources directory exists
            File::ensureDirectoryExists($this->getResourcePath(), 0777);

            // Download the database archive
            $client = new Client(["timeout" => 30]);
            $client->get($this->getDatabaseDownloadLink(), [
                "sink" => $this->getDatabaseArchivePath(),
            ]);

            // Extract the MMDB file from the archive
            $this->extractDatabaseFromArchive();
        } catch (GeoIpException $exception) {
            throw $exception;
        } catch (\Throwable $exception) {
        }
    }

    /**
     * Extract the first .mmdb file from the archive
     */
    private function extractDatabaseFromArchive(): void
    {
        // Extract the downloaded database archive
        $archive = new PharData($this->getDatabaseArchivePath());
        $archive->extractTo($path = $this->getResourcePath("/maxmind"), null, true);

        /** @var \SplFileInfo[] $finder */
        $finder = Finder::create()->files()->in($path)->name("*.mmdb");
        foreach ($finder as $file) {
            File::move($file->getRealPath(), $this->getDatabasePath());

            break;
        }
    }

    /**
     * @return bool
     */
    private function shouldDownloadDatabase(): bool
    {
        $interval     = 7 * 24 * 3600; // 7 Days
        $databasePath = $this->getDatabasePath();

        return !File::exists($databasePath)
            || time() - filemtime($databasePath) > $interval;
    }

    /**
     * Delete the junk files
     */
    private function cleanUpDatabaseResources(): void
    {
        File::delete($this->getDatabaseArchivePath());
        File::deleteDirectory($this->getResourcePath("/maxmind"));
    }

    /**
     * @return string
     * @throws GeoIpException
     */
    private function getDatabaseDownloadLink(): string
    {
        $link = $this->config("link");

        return $link ?? $this->buildDatabaseDownloadLink();
    }

    /**
     * @return string
     * @throws GeoIpException
     */
    private function buildDatabaseDownloadLink(): string
    {
        $license = $this->config("license");
        if (empty($license)) {
            throw new GeoIpException("No license key is provided for MaxMind provider.");
        }

        return sprintf("https://download.maxmind.com/app/geoip_download?edition_id=GeoLite2-Country&license_key=%s&suffix=tar.gz", $license);
    }

    /**
     * @param string|null $name
     * @return string
     */
    private function getResourcePath(?string $name = null): string
    {
        return storage_path("app/geoip/{$name}");
    }

    /**
     * @return string
     */
    private function getDatabasePath(): string
    {
        return $this->getResourcePath("/maxmind.mmdb");
    }

    /**
     * @return string
     */
    private function getDatabaseArchivePath(): string
    {
        return $this->getResourcePath("/maxmind.tar.gz");
    }
}
