<?php

declare(strict_types = 1);

namespace JuicyCodes\Core;

use Carbon\Carbon;
use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Request;
use JuicyCodes\Core\Support\InvalidResponse;

class License
{
    private object $data;

    private Encrypter $encrypter;

    public function __construct(private Product $product)
    {
        $this->encrypter = new Encrypter(
            "fg6mDgIz66d1YPedpleHJvwFOLZyxDoN",
            "AES-256-CBC"
        );
    }

    public function isValid(): bool
    {
        return $this->data()->valid === true;
    }

    public function verify(bool $showReasons = false): void
    {
        if ($this->isValid()) {
            return;
        }

        if (!Request::expectsJson()) {
            InvalidResponse::htmlResponse();
        }

        InvalidResponse::jsonResponse($this->data()->resons, $showReasons);
    }

    public function data(): object
    {
        if (!isset($this->data)) {
            $this->data = $this->getCachedLicenseData();
        }

        // Prevent using license data older then 4 hours
        $timestamp = Carbon::parse($this->data->timestamp);
        if ($timestamp->addHours(4)->isPast()) {
            $this->forgetCache();

            return $this->data();
        }

        return $this->data;
    }

    public function forgetCache(): void
    {
        unset($this->data);
        Cache::forget($this->getCacheKey());
    }

    private function getCachedLicenseData(): object
    {
        $callback = fn(): string => $this->encrypter->encrypt($this->sendRequest());
        $response = Cache::remember($this->getCacheKey(), now()->addHours(4), $callback);

        return json_decode($this->encrypter->decrypt($response));
    }

    private function sendRequest(): string
    {
        # Generate request endpoint & initiate cURL session
        $curl = curl_init(sprintf(
            "https://license.juicycodes.net/v3/verify/%s/%s",
            $this->product->getCodename(),
            App::licenseKey()
        ));

        // Set request options
        curl_setopt_array($curl, [
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $this->getRequiredData(),
            CURLOPT_CONNECTTIMEOUT => 15,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
        ]);

        // Execute cURL request & get information
        $response    = (string) curl_exec($curl);
        $information = curl_getinfo($curl);

        // Close the cURL session
        curl_close($curl);

        // Check if request was successful
        if ($information["http_code"] === 200) {
            // Check if received response in expected format & not empty
            if ($this->isValidResponse($response, $information)) {
                return $response;
            } else {
                $reason = "Empty or unrecognizable response received!";
            }
        } else {
            $reason = sprintf("Unknown error occurred (STATUS: %s)!", $information["http_code"]);
        }

        return json_encode([
            "valid"     => false,
            "reasons"   => [$reason],
            "timestamp" => Carbon::now(),
        ]);
    }

    private function getRequiredData(): array
    {
        return [
            "root_link"    => url("/"),
            "root_path"    => base_path(),
            "app_version"  => $this->product->getVersion(),
            "data_payload" => base64_encode(json_encode([
                "inputs" => Request::all(),
                "server" => Request::server(),
            ])),
        ];
    }

    private function isValidResponse(string $response, array $information): bool
    {
        return !empty($response)
            && $information["content_type"] === "application/json"
            && $this->isValidJson($response);
    }

    private function getCacheKey(): string
    {
        return sprintf("license:%s", $this->product->getCodeName());
    }

    private function isValidJson(string $value): bool
    {
        try {
            $data = json_decode($value, true, 512, JSON_THROW_ON_ERROR);

            return !empty($data);
        } catch (\JsonException) {
            return false;
        }
    }
}
