<?php

declare(strict_types=1);

namespace JuicyCodes\Core\Support;

use Illuminate\Support\Carbon;
use JuicyCodes\Core\Data\License\Responses\LicenseResponse;
use JuicyCodes\Core\Data\License\Responses\UnverifiedResponse;
use JuicyCodes\Core\Data\License\Responses\VerifiedResponse;
use Throwable;

final readonly class LicenseResponseBuilder
{
    public static function build(HttpRequest $request): LicenseResponse
    {
        try {
            return self::buildResponse($request);
        } catch (Throwable) {
            return self::buildUnverifiedResponse('Failed to parse response!');
        }
    }

    private static function buildResponse(HttpRequest $request): LicenseResponse
    {
        if (! in_array($request->information('http_code'), [200, 400])) {
            return self::buildUnverifiedResponse(sprintf(
                'Unexpected http response received: HTTP%s!',
                $request->information('http_code')
            ));
        }

        if (! $request->isValidJsonResponse()) {
            return self::buildUnverifiedResponse('Empty or unrecognizable response received!');
        }

        $response = $request->json();

        return match ($request->information('http_code') === 200) {
            true => VerifiedResponse::from($response),
            default => self::buildUnverifiedResponse($response),
        };
    }

    private static function buildUnverifiedResponse(object|string $response): UnverifiedResponse
    {
        $error = match (is_string($response)) {
            true => $response,
            false => $response->error ?? 'Unknown error occurred!',
        };

        return UnverifiedResponse::from([
            'error' => $error,
            'timestamp' => Carbon::parse($response->timestamp ?? now()),
        ]);
    }
}
