<?php

declare(strict_types=1);

namespace JuicyCodes\Core\Support;

use Illuminate\Support\Arr;

final class HttpRequest
{
    protected string $response;

    /**
     * @var array<string, mixed>
     */
    protected array $information = [];

    /**
     * @param  array<string,string|null>  $fields
     */
    public function send(string $url, array $fields = []): string
    {
        // Initialize a cURL session
        $curl = curl_init();

        // Set request options
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $fields,
            CURLOPT_CONNECTTIMEOUT => 15,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_USERAGENT => 'Curl/PHP',
            CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
        ]);

        // Execute cURL request & get information
        $this->response = (string) curl_exec($curl);
        $this->information = curl_getinfo($curl);

        // Close the cURL session
        curl_close($curl);

        return $this->response;
    }

    public function response(): string
    {
        return $this->response;
    }

    public function json(): object
    {
        return json_decode($this->response());
    }

    /**
     * @return mixed|array<string, mixed>
     */
    public function information(?string $key = null): mixed
    {
        return Arr::get($this->information, $key);
    }

    public function isValidJsonResponse(): bool
    {
        return str_starts_with($this->information['content_type'], 'application/json')
            && ! empty($this->response)
            && json_validate($this->response);
    }
}
