<?php

declare(strict_types=1);

namespace JuicyCodes\Core;

use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Request;
use JuicyCodes\Core\Data\License\Responses\LicenseResponse;
use JuicyCodes\Core\Support\HttpRequest;
use JuicyCodes\Core\Support\LicenseResponseBuilder;
use JuicyCodes\Core\Support\LicenseVerificationFailedResponse;

final class LicenseChecker
{
    private readonly Encrypter $encrypter;

    private LicenseResponse $response;

    public function __construct(
        readonly protected Product $product,
    ) {
        $this->encrypter = new Encrypter(
            key: 'APuVsinPkGLPJrfxMMBQkNqwW8gV7ADG',
            cipher: 'AES-256-CBC'
        );
    }

    public function verifyOrDie(?bool $json = null): void
    {
        if ($this->isValid()) {
            return;
        }

        match ($json ?? Request::expectsJson()) {
            true => LicenseVerificationFailedResponse::json(),
            default => LicenseVerificationFailedResponse::html(),
        };
    }

    public function isValid(): bool
    {
        return $this->response()->isValid();
    }

    public function response(): LicenseResponse
    {
        $this->response ??= $this->cachedResponse();

        // Make sure that response is not older than 4 hours
        if ($this->response->timestamp->addHours(4)->isPast()) {
            //@codeCoverageIgnoreStart
            return $this->clearCache()->response();
            //@codeCoverageIgnoreEnd
        }

        return $this->response;
    }

    public function clearCache(): self
    {
        unset($this->response);
        Cache::forget($this->getCacheKey());

        return $this;
    }

    private function cachedResponse(): LicenseResponse
    {
        // Cache the license response for 4 hours
        $cached = Cache::remember(
            key: $this->getCacheKey(),
            ttl: now()->addHours(4),
            callback: fn () => $this->encrypter->encrypt($this->sendCheckRequest())
        );

        return $this->encrypter->decrypt($cached);
    }

    private function sendCheckRequest(): LicenseResponse
    {
        $request = new HttpRequest;
        $request->send($this->buildApiUrl(), [
            'root_link' => url('/'),
            'root_path' => base_path(),
            'app_version' => $this->product->version,
            'core_origin' => App::license()->coreOrigin,
            'data_payload' => base64_encode((string) json_encode([
                'inputs' => Request::all(),
                'server' => Request::server(),
            ])),
        ]);

        return LicenseResponseBuilder::build($request);
    }

    private function buildApiUrl(): string
    {
        $endpoint = match (App::license()->coreOrigin === null) {
            true => 'https://license.juicycodes.net/v4/verify/%s/%s',
            false => 'https://license.juicycodes.net/v4/verify/%s/%s/remote',
        };

        return sprintf($endpoint, App::license()->key, $this->product->codename);
    }

    private function getCacheKey(): string
    {
        return hash('xxh128', "license:{$this->product->codename}");
    }
}
