<?php

declare(strict_types = 1);

namespace JuicyCodes\Core;

use Carbon\Carbon;
use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Request;
use JuicyCodes\Core\Support\InvalidResponse;
use function DeepCopy\deep_copy;

final class License
{
    private string $key;

    private object $license;

    private AppRepository $repo;

    private Encrypter $encrypter;

    public function __construct(AppRepository $app, string $key)
    {
        $this->repo = $app;
        $this->setKey($key);
        $this->encrypter = $this->encrypterInstance();
    }

    public function getKey(): string
    {
        return $this->key;
    }

    public function setKey(string $key): void
    {
        $this->key = $key;
    }

    public function isValid(string $codename = null): bool
    {
        if ($codename !== null && $this->license()->valid) {
            return $this->findChildByCodename($codename) !== null;
        }

        return $this->license()->valid;
    }

    public function getDetails(string $codename = null): object
    {
        /** @var object $license */
        $license = deep_copy($this->license());

        if ($codename !== null && $this->license()->valid) {
            $child = $this->findChildByCodename($codename);

            if ($child === null) {
                return InvalidResponse::childNotFoundResponse($codename);
            }

            // Remove the child products list
            unset($license->properties->children);

            // Overwrite the main products limit with the child one
            $license->properties->limit = $child->limit;
        }

        return $license;
    }

    public function verify(string $codename = null): void
    {
        if ($this->isValid($codename)) {
            return;
        }

        if (Request::expectsJson()) {
            InvalidResponse::jsonResponse();
        }

        InvalidResponse::htmlResponse();
    }

    private function license(): object
    {
        if (!isset($this->license)) {
            $this->license = json_decode($this->encrypter->decrypt(
                $this->getCachedLicenseData()
            ));
        }

        // Prevent using license data older then 2 hours
        $timestamp = Carbon::parse($this->license->timestamp);
        if ($timestamp->addHours(2)->isPast()) {
            unset($this->license);
            Cache::forget("license");
            return $this->license();
        }

        return $this->license;
    }


    private function getCachedLicenseData(): string
    {
        $callback = fn(): string => $this->encrypter->encrypt($this->sendRequest());

        return Cache::remember("license", now()->addHours(2), $callback);
    }

    private function sendRequest(): string
    {
        # Generate request endpoint & initiate cURL session
        $curl = curl_init(
            "https://license.juicycodes.net" .
            "/v3/verify/{$this->repo->app()->getCodename()}/{$this->getKey()}"
        );

        # Set request options
        curl_setopt_array($curl, [
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $this->getRequiredData(),
            CURLOPT_CONNECTTIMEOUT => 15,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
        ]);

        # Execute cURL request & get information
        $response    = (string) curl_exec($curl);
        $information = curl_getinfo($curl);

        # Close the cURL session
        curl_close($curl);

        # Check if request was successful
        if ($information["http_code"] === 200) {
            # Check if received response in expected format & not empty
            if ($this->isValidResponse($response, $information)) {
                return $response;
            } else {
                $reason = "Empty or unrecognizable response received!";
            }
        } else {
            $reason = sprintf("Unknown error occurred (STATUS: %s)!", $information["http_code"]);
        }

        return json_encode([
            "valid"     => false,
            "reasons"   => [$reason],
            "timestamp" => Carbon::now(),
        ]);
    }

    private function encrypterInstance(): Encrypter
    {
        return new Encrypter("fg6mDgIz66d1YPedpleHJvwFOLZyxDoN", "AES-256-CBC");
    }

    private function findChildByCodename(string $codename): ?object
    {
        return Arr::first(
            $this->license()->properties->children,
            fn(object $child) => $child->code === $codename
        );
    }

    private function getRequiredData(): array
    {
        return [
            "root_link"    => url("/"),
            "root_path"    => base_path(),
            "app_version"  => $this->repo->app()->getVersion(),
            "data_payload" => base64_encode(json_encode([
                "inputs" => Request::all(),
                "server" => Request::server(),
            ])),
        ];
    }

    private function isValidResponse(string $response, array $information): bool
    {
        return !empty($response)
            && $information["content_type"] === "application/json"
            && $this->isValidJson($response);
    }

    private function isValidJson(string $value): bool
    {
        try {
            $data = json_decode($value, true, 512, JSON_THROW_ON_ERROR);

            return !empty($data);
        } catch (\JsonException) {
            return false;
        }
    }
}
