<?php

declare(strict_types=1);

namespace Bridge\Domains\Item\Models;

use Bridge\Database\Factories\ItemPriceFactory;
use Bridge\Domains\Currency\Currency;
use Bridge\Support\Casts\Amount;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * @property-read Currency $buyCurrency
 */
class ItemPrice extends Model
{
    use HasFactory;

    protected $casts = [
        'buy_for' => Amount::class,
        'sell_for' => Amount::class,
    ];

    /**
     * @return Attribute<float,null|float>
     */
    public function buyFor(): Attribute
    {
        return Attribute::set(fn (?float $value) => $value ?? 0.00);
    }

    /**
     * @return Attribute<float,null|float>
     */
    public function sellFor(): Attribute
    {
        return Attribute::set(fn (?float $value) => $value ?? 0.00);
    }

    /**
     * @return BelongsTo<Item,ItemPrice>
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * @return BelongsTo<Currency,ItemPrice>
     */
    public function buyCurrency(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'buy_currency_id');
    }

    /**
     * @return BelongsTo<Currency,ItemPrice>
     */
    public function sellCurrency(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'sell_currency_id');
    }

    /**
     * @return Factory<ItemPrice>
     */
    protected static function newFactory(): Factory
    {
        return ItemPriceFactory::new();
    }
}
