<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Currency\Currency;
use Bridge\Domains\Invoice\Enums\InvoiceContactType;
use Bridge\Domains\Invoice\Enums\InvoiceStatus;
use Bridge\Domains\Invoice\Enums\InvoiceType;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Invoice\Models\InvoiceItem;
use Bridge\Domains\Invoice\Models\InvoiceSchedule;
use Bridge\Domains\Member\Customer;
use Bridge\Domains\Supplier\Supplier;
use Closure;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<Invoice>
 */
class InvoiceFactory extends Factory
{
    protected $model = Invoice::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'notes' => fake()->optional(0.4)->sentence(),
            'type' => fake()->randomElement(InvoiceType::cases()),
            'due_on' => fake()->dateTimeBetween('now', '+1 week')->format('Y-m-d'),
            'issued_on' => fake()->dateTimeBetween('-1 week', 'now')->format('Y-m-d'),

            'status' => $this->generateStatus(),
            'contact_id' => $this->getContactFactory(),
            'contact_type' => $this->getContactType(),
            'currency_id' => Currency::factory(),
            'currency_rate' => $this->getCurrencyRate(),

            'total' => $total = fake()->randomFloat(2, 10, 1000),
            'subtotal' => $total,
        ];
    }

    /**
     * @return Factory<Invoice>
     */
    public function withItems(): Factory
    {
        return $this->afterCreating(function (Invoice $invoice) {
            // Add some items to the invoice
            InvoiceItem::factory(rand(1, 5))->for($invoice)->create();

            // Calculate the sub-total in the invoice's currency
            $subTotal = $invoice->items->sum(function (InvoiceItem $item) use ($invoice) {
                $amountInUsd = $item->quantity * ($item->price / $item->currency_rate);

                return $amountInUsd * $invoice->currency_rate;
            });

            // Update the invoice
            $invoice->total = $subTotal;
            $invoice->subtotal = $subTotal;
            $invoice->save();
        });
    }

    /**
     * @return Factory<Invoice>
     */
    public function withSchedule(): Factory
    {
        return $this
            ->recurring()
            ->afterCreating(function (Invoice $invoice) {
                InvoiceSchedule::factory()->for($invoice)->create();
            });
    }

    /**
     * @return Factory<Invoice>
     */
    public function sale(): Factory
    {
        return $this->state(fn () => [
            'type' => InvoiceType::Sale,
        ]);
    }

    /**
     * @return Factory<Invoice>
     */
    public function purchase(): Factory
    {
        return $this->state(fn () => [
            'type' => InvoiceType::Purchase,
        ]);
    }

    /**
     * @return Factory<Invoice>
     */
    public function recurring(): Factory
    {
        return $this->state(fn () => [
            'type' => fake()->randomElement([
                InvoiceType::RecurringSale,
                InvoiceType::RecurringPurchase,
            ]),
        ]);
    }

    private function generateStatus(): Closure
    {
        return function (array $attributes) {
            /** @var InvoiceType $type */
            $type = match (is_string($attributes['type'])) {
                false => $attributes['type'],
                true => InvoiceType::from($attributes['type']),
            };

            $status = fake()->randomElement(match ($type->isRecurring()) {
                false => InvoiceStatus::base(),
                true => InvoiceStatus::recurring(),
            });

            return $status->value;
        };
    }

    private function getContactFactory(): Closure
    {
        return function (array $attributes) {
            return $attributes['type']->isSale()
                ? Customer::factory()
                : Supplier::factory();
        };
    }

    private function getContactType(): Closure
    {
        return function (array $attributes) {
            return InvoiceContactType::fromInvoiceType(
                is_string($attributes['type'])
                    ? InvoiceType::from($attributes['type'])
                    : $attributes['type']
            )->value;
        };
    }

    private function getCurrencyRate(): Closure
    {
        return function (array $attributes) {
            return Currency::query()
                ->where('id', $attributes['currency_id'])
                ->sole()
                ->rate;
        };
    }
}
