<?php

declare(strict_types=1);

namespace Bridge\Support;

use Carbon\CarbonImmutable;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Date;

class SequentialCodeGenerator
{
    /**
     * @template TModel of Model
     *
     * @param  Builder<TModel>  $query
     * @param  string[]  $data
     *
     * @throws Exception
     */
    public static function generate(Builder $query, array $data, ?CarbonImmutable $date = null): string
    {
        $date ??= Date::now()->toImmutable();

        $codePrefix = strtoupper(implode('', [
            ...$data,
            $date->format('y'), // Last two digits of year
            chr($date->month + 64), // Month as letter (A to L)
            $date->format('d'), // Day of month
        ]));

        // Count existing records with the same prefix
        $existingRecords = $query
            ->where('number', 'like', $codePrefix.'%')
            ->count();

        // Generate a potential sequential code
        $sequentialCode = $codePrefix.'A0';

        // Increment the sequential code for each existing record
        for ($i = 1; $i <= $existingRecords; $i++) {
            $sequentialCode = str_increment($sequentialCode);
        }

        // If the invoice number is too long or already exists, throw an exception
        if (strlen($sequentialCode) > 15 || $query->where('number', $sequentialCode)->exists()) {
            $model = get_class($query->getModel());
            throw new Exception("Unable to generate sequential code for [{$model}]");
        }

        return $sequentialCode;
    }
}
