<?php

declare(strict_types=1);

namespace Bridge\Support;

use Closure;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class RandomNoun
{
    public static function pick(): string
    {
        $nouns = Cache::rememberForever('nouns', fn () => (new self)->nouns());

        return Str::upper($nouns[array_rand($nouns)]);
    }

    public static function uniquePick(Closure $callback, int $maxAttempts = 10): ?string
    {
        do {
            $noun = self::pick();

            if ($maxAttempts-- <= 0) {
                return null;
            }
        } while ($callback($noun) === false);

        return $noun;
    }

    /**
     * @return list<string>
     */
    protected function nouns(): array
    {
        if (app()->runningUnitTests()) {
            return ['NOUN1', 'NOUN2', 'NOUN3'];
        }

        $wordList = Http::timeout(5)
            ->get('https://www.desiquintans.com/downloads/nounlist/nounlist.txt')
            ->body();

        $badWords = $this->badWords();
        $nounList = self::basicWordFilter($wordList);

        return array_diff($nounList, $badWords);
    }

    /**
     * @return list<string>
     */
    protected function badWords(): array
    {
        if (app()->runningUnitTests()) {
            return [];
        }

        $wordList = Http::timeout(5)
            ->get('https://www.cs.cmu.edu/~biglou/resources/bad-words.txt')
            ->body();

        return self::basicWordFilter($wordList);
    }

    /**
     * @return list<string>
     */
    private function basicWordFilter(string $wordList): array
    {
        return collect(explode("\n", $wordList))
            ->reject(function (string $noun) {
                return strlen($noun) > 5
                    || strlen($noun) < 3
                    || ctype_upper($noun)
                    || preg_match('/[^a-z]/', $noun);
            })
            ->toArray();
    }
}
