<?php

declare(strict_types=1);

namespace Bridge\Support\Pagination;

use Bridge\Support\MaskedId\MaskedId;
use Bridge\Support\Pagination\Data\PaginatedRecordsData;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Str;

/**
 * @template TModel of Model
 * @template TData of DataWithPaginator
 */
class PaginatedRecordsTransformer
{
    /**
     * @param  LengthAwarePaginator<TModel>  $paginator
     * @return PaginatedRecordsData<TData>
     */
    public static function transform(Builder $subject, LengthAwarePaginator $paginator): PaginatedRecordsData
    {
        return PaginatedRecordsData::from([
            'data' => $paginator->items(),
            'pagination' => [
                'totalRecords' => $paginator->total(),
                'recordsFrom' => $paginator->firstItem() ?? 0,
                'recordsTo' => $paginator->lastItem() ?? 0,

                'totalPages' => $paginator->lastPage(),
                'currentPage' => $paginator->currentPage(),
                'linkPrefix' => self::buildPaginationLink($paginator),
            ],
            'meta' => [
                'sort' => request()->query('sort'),
                'filters' => self::getFilterParams(),
                'recordsExists' => $subject->exists(),
            ],
        ]);
    }

    /**
     * @param  LengthAwarePaginator<TModel>  $records
     */
    private static function buildPaginationLink(Paginator $records): string
    {
        $url = $records->withQueryString()->url(PHP_INT_MAX);

        return Str::beforeLast($url, (string) PHP_INT_MAX);
    }

    /**
     * @return null|array<string, string|string[]>
     */
    private static function getFilterParams(): ?array
    {
        $filters = request()->query('filter');

        if (!is_array($filters)) {
            return null;
        }

        return collect($filters)
            ->map(function ($value, string $key) {
                if ($key === 'search') {
                    return $value;
                }

                if (MaskedId::canBeEncoded($value, $key)) {
                    return app(MaskedId::class)->encodeModelId(Model::class, $value);
                }

                return explode(',', (string) $value);
            })
            ->toArray();
    }
}
