<?php

declare(strict_types=1);

namespace Bridge\Support\MaskedId;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\MissingAttributeException;
use RuntimeException;

trait HasMaskedId
{
    public function getRouteKey(): mixed
    {
        if ($this->getKeyType() === 'int') {
            return $this->maskModelId($this->getKey());
        }

        throw new RuntimeException('Key should be of type int to encode into a fake id.');
    }

    /** @phpstan-ignore-next-line */
    public function resolveRouteBindingQuery($query, $value, $field = null): mixed
    {
        if (MaskedId::canBeDecoded($value)) {
            $value = app(MaskedId::class)->decodeModelId($value);
        }

        return parent::resolveRouteBindingQuery($query, $value, $field);
    }

    protected function initializeHasMaskedId(): void
    {
        $this->makeHidden('id')
            ->append('masked_id');
    }

    /**
     * @return Attribute<int,never>
     */
    protected function maskedId(): Attribute
    {
        return Attribute::get(function () {
            try {
                if ($this->id === null) {
                    throw new MissingAttributeException($this, 'id');
                }

                return $this->maskModelId($this->id);
            } catch (MissingAttributeException) {
                return 0;
            }
        });
    }

    protected function maskModelId(int $value): int
    {
        return app(MaskedId::class)
            ->encodeModelId(static::class, $value);
    }
}
