<?php

declare(strict_types=1);

namespace Bridge\Domains\Server\Models;

use Bridge\Database\Factories\ItemServerDetailsFactory;
use Bridge\Domains\Member\Customer;
use Bridge\Domains\Server\Data\ServerCredentialData;
use Bridge\Domains\Server\Support\CredentialEncrypter;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Collection;

class ItemServerDetails extends Model
{
    use HasFactory;

    protected $casts = [
        'due_within' => 'int',
        'invoice_date' => 'date',
    ];

    /**
     * @return Attribute<Collection<int,ServerCredentialData>,string>
     */
    public function credentials(): Attribute
    {
        return Attribute::make(
            get: fn ($data) => app(CredentialEncrypter::class)->decrypt($data),
            set: fn ($data) => app(CredentialEncrypter::class)->encrypt($data)
        );
    }

    /**
     * @return BelongsTo<Server, ItemServerDetails>
     */
    public function server(): BelongsTo
    {
        return $this->belongsTo(Server::class, 'item_id');
    }

    /**
     * @return BelongsTo<Customer, ItemServerDetails>
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class, 'customer_id');
    }

    /**
     * @return Factory<ItemServerDetails>
     */
    protected static function newFactory(): Factory
    {
        return ItemServerDetailsFactory::new();
    }
}
