<?php

declare(strict_types=1);

namespace Bridge\Domains\Item\Models;

use Bridge\Database\Factories\ItemFactory;
use Bridge\Domains\Item\Enums\ItemStatus;
use Bridge\Domains\Item\Enums\ItemType;
use Bridge\Domains\Supplier\Supplier;
use Bridge\Support\Data\Data;
use Bridge\Support\ModelTraits;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @property-read ItemPrice $price
 */
class Item extends Model
{
    use HasFactory;
    use ModelTraits;
    use SoftDeletes;

    protected $table = 'items';

    protected $casts = [
        'type' => ItemType::class,
        'status' => ItemStatus::class,
    ];

    protected static function booted(): void
    {
        if (static::class !== self::class) {
            static::addGlobalScope('type', function (Builder $query) {
                $query->where('type', ItemType::fromModel(static::class));
            });
        }
    }

    /**
     * @return HasOne<ItemPrice>
     */
    public function price(): HasOne
    {
        return $this->hasOne(ItemPrice::class, 'item_id');
    }

    /**
     * @return BelongsTo<Supplier,Item>
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * @return HasMany<ItemMeta>
     */
    public function meta(): HasMany
    {
        return $this->hasMany(ItemMeta::class, 'item_id');
    }

    /**
     * @param  Builder<static>  $query
     * @return Builder<static>
     */
    public function scopeActive(Builder $query): Builder
    {
        return $query->where('status', ItemStatus::Active);
    }

    /**
     * @param  Builder<static>  $query
     * @return Builder<static>
     */
    public function scopeWithViewRelations(Builder $query): Builder
    {
        return $query->with(['meta', 'supplier', 'price', 'price.buyCurrency', 'price.sellCurrency']);
    }

    /**
     * @param  Builder<static>  $query
     * @return Builder<static>
     */
    public function scopeWhereMeta(Builder $query, string $Key, mixed $value): Builder
    {
        return $query->whereHas('meta', fn (Builder $query) => $query->where('key', $Key)->where('value', $value));
    }

    public function loadViewRelations(): static
    {
        return $this->load(['meta', 'supplier', 'price', 'price.buyCurrency', 'price.sellCurrency']);
    }

    public function loadFormRelations(): static
    {
        return $this->load(['meta', 'price']);
    }

    /**
     * @param  array<string,int|string>|Data  $meta
     */
    public function setMetaData(array|Data $meta): void
    {
        $meta = $meta instanceof Data
            ? $meta->toArray() : $meta;

        foreach ($meta as $key => $value) {
            $this->meta()->updateOrCreate(
                compact('key'),
                compact('value')
            );
        }
    }

    public function getMetaData(string $key): mixed
    {
        $value = $this->meta()->where('key', $key)->value('value');

        if ($value === null) {
            throw new \Exception("Meta data with key: {$key} does not exist");
        }

        return $value;
    }

    public function isInactive(): bool
    {
        return $this->status === ItemStatus::Inactive;
    }

    /**
     * @return Factory<Item>
     */
    protected static function newFactory(): Factory
    {
        return ItemFactory::new();
    }
}
