<?php

declare(strict_types=1);

namespace Bridge\Domains\Invoice\Tasks;

use Bridge\Domains\Invoice\Enums\InvoiceStatus;
use Bridge\Domains\Invoice\Enums\InvoiceType;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Invoice\Models\InvoiceItem;
use Bridge\Domains\License\Models\License;
use Bridge\Domains\License\Models\Product;
use Bridge\Domains\License\Support\GenerateLicenseKey;
use Bridge\Domains\Member\Customer;

class MarkInvoiceAsPaid extends AbstractTask
{
    /**
     * @throws \Throwable
     */
    protected static function handle(Invoice $invoice): void
    {
        $invoice->loadMissing('items', 'items.item', 'contact', 'generator');

        // Mark the invoice as paid
        $invoice->status = InvoiceStatus::Paid;
        $invoice->save();

        // Clear all licenses associated with this invoice
        self::cleanupItemLicenses($invoice);

        // Clear all item ownership associated with this invoice
        self::cleanupItemOwnership($invoice);

        // Assign ownership of items based on this invoice
        if (self::shouldItemBeAssigned($invoice)) {
            self::assignInvoiceItemsOwnership(self::getTargetInvoice($invoice));
            self::createLicenseForInvoiceItems(self::getTargetInvoice($invoice));
        }
    }

    protected static function assignInvoiceItemsOwnership(Invoice $invoice): void
    {
        $invoice->memberItems()->createMany(
            $invoice->items
                ->reject(fn (InvoiceItem $item) => $item->item->type->isServer())
                ->map(fn (InvoiceItem $item) => [
                    'item_id' => $item->item_id,
                    'quantity' => $item->quantity,
                    'member_id' => $invoice->contact->id,
                ])
        );
    }

    private static function createLicenseForInvoiceItems(Invoice $invoice): void
    {
        // Don't do anything if the invoice was generated by a recurring invoice
        if ($invoice->type->isRecurring()) {
            return;
        }

        $itemIds = $invoice->items->pluck('item_id');
        $products = Product::query()->isCore()->whereIn('erp_product_id', $itemIds)->get();

        $products->each(function (Product $product) use ($invoice) {
            // Get the quantity of the product in the invoice
            $quantity = $invoice->items
                ->firstOrFail('item_id', $product->erp_product_id)
                ->quantity;

            for ($i = 0; $i < $quantity; $i++) {
                // Create the license
                $license = License::query()->create([
                    'key' => GenerateLicenseKey::unique(),
                    'customer_id' => $invoice->contact->id,
                ]);

                // Attach the core product
                $license->licensed()->create([
                    'product_id' => $product->id,
                    'invoice_id' => $invoice->id,
                    'updates_till' => now('UTC')->addMonths(6),
                ]);
            }
        });
    }

    private static function shouldItemBeAssigned(Invoice $invoice): bool
    {
        $isInvoiceGeneratorAllowed = match (true) {
            $invoice->generator === null => true,
            $invoice->generator instanceof Invoice => true,
            default => false,
        };

        return $isInvoiceGeneratorAllowed
            && $invoice->contact instanceof Customer
            && $invoice->type === InvoiceType::Sale;
    }
}
