<?php

declare(strict_types=1);

namespace Bridge\Domains\Invoice\Tasks;

use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\License\Models\License;
use Illuminate\Support\Facades\DB;

abstract class AbstractTask
{
    abstract protected static function handle(Invoice $invoice): void;

    public static function run(Invoice $invoice): void
    {
        DB::transaction(fn () => static::handle($invoice));
    }

    protected static function cleanupPayments(Invoice $invoice): void
    {
        $invoice->payments()->delete();
    }

    protected static function cleanupItemOwnership(Invoice $invoice): void
    {
        self::getTargetInvoice($invoice)->memberItems()->delete();
    }

    protected static function cleanupItemLicenses(Invoice $invoice): void
    {
        // Don't do anything if the invoice was generated by a recurring invoice
        if (self::getTargetInvoice($invoice)->type->isRecurring()) {
            return;
        }

        $query = License::query()
            ->whereHas('coreProduct', function ($query) use ($invoice) {
                return $query->where('invoice_id', $invoice->id);
            });

        $query->update(['message' => 'Auto deleted due to invoice status change']);
        $query->delete();
    }

    protected static function getTargetInvoice(Invoice $invoice): Invoice
    {
        if ($invoice->generator instanceof Invoice && $invoice->generator->type->isRecurring()) {
            return $invoice->generator;
        }

        return $invoice;
    }
}
