<?php

declare(strict_types=1);

namespace Bridge\Domains\Invoice\Models;

use Bridge\Database\Factories\InvoiceFactory;
use Bridge\Domains\Currency\Currency;
use Bridge\Domains\Discount\Enums\DiscountType;
use Bridge\Domains\Invoice\Enums\InvoiceStatus;
use Bridge\Domains\Invoice\Enums\InvoiceType;
use Bridge\Domains\Invoice\Support\Traits\GenerateInvoiceNumber;
use Bridge\Domains\Invoice\Support\Traits\InvoiceCommonCalculations;
use Bridge\Domains\Member\Models\MemberItem;
use Bridge\Support\Casts\Amount;
use Bridge\Support\Casts\Rate;
use Bridge\Support\ModelTraits;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @property-read Currency $currency
 */
class Invoice extends Model
{
    use GenerateInvoiceNumber;
    use HasFactory;
    use InvoiceCommonCalculations;
    use ModelTraits;
    use SoftDeletes;

    protected $casts = [
        'fee' => Amount::class,
        'total' => Amount::class,
        'subtotal' => Amount::class,
        'discount' => Amount::class,
        'currency_rate' => Rate::class,

        'due_on' => 'date:Y-m-d',
        'issued_on' => 'date:Y-m-d',

        'type' => InvoiceType::class,
        'status' => InvoiceStatus::class,
        'discount_type' => DiscountType::class,
    ];

    /**
     * @return MorphTo<Model,Invoice>
     */
    public function contact(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * @return HasMany<InvoiceItem>
     */
    public function items(): HasMany
    {
        return $this->hasMany(InvoiceItem::class)
            ->orderBy('order');
    }

    /**
     * @return HasMany<InvoicePayment>
     */
    public function payments(): HasMany
    {
        return $this->hasMany(InvoicePayment::class);
    }

    /**
     * @return HasMany<MemberItem>
     */
    public function memberItems(): HasMany
    {
        return $this->hasMany(MemberItem::class);
    }

    /**
     * @return BelongsTo<Currency,Invoice>
     */
    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    /**
     * @return MorphTo<Model,Invoice>
     */
    public function generator(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * @return Factory<Invoice>
     */
    protected static function newFactory(): Factory
    {
        return InvoiceFactory::new();
    }
}
