<?php

declare(strict_types=1);

namespace Bridge\Domains\Discount\Support;

use Bridge\Domains\Discount\Enums\DiscountCustomerType;
use Bridge\Domains\Discount\Enums\DiscountStatus;
use Bridge\Domains\Discount\Models\Discount;
use Bridge\Domains\Invoice\Enums\InvoiceContactType;
use Bridge\Domains\Invoice\Enums\InvoiceStatus;
use Bridge\Domains\Invoice\Models\Invoice;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class DiscountValidator
{
    public function __construct(
        readonly public int $discountId,
        readonly public int $customerId,
    ) {}

    public static function isValid(int $discountId, int $customerId): bool
    {
        return (new self($discountId, $customerId))->validate();
    }

    public function validate(): bool
    {
        try {
            $discount = self::fetchOrFail($this->discountId);
        } catch (ModelNotFoundException) {
            return false;
        }

        return match ($discount->customer_type) {
            DiscountCustomerType::All => true,
            DiscountCustomerType::Old => $this->customerHasPaidInvoices(),
            DiscountCustomerType::New => !$this->customerHasPaidInvoices(),
        };
    }

    public static function fetchOrFail(?int $id = null): Discount
    {
        $query = fn () => Discount::query()
            ->where('status', DiscountStatus::Active)
            ->where('id', $id)
            ->firstOrFail();

        return once($query);
    }

    private function customerHasPaidInvoices(): bool
    {
        return Invoice::query()
            ->where('status', InvoiceStatus::Paid)
            ->where('contact_id', $this->customerId)
            ->where('contact_type', InvoiceContactType::Customer->value)
            ->exists();
    }
}
