<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Currency\Currency;
use Bridge\Domains\Item\Enums\ItemType;
use Bridge\Domains\Item\Models\Item;
use Bridge\Domains\Item\Models\ItemPrice;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\ModelNotFoundException;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<ItemPrice>
 */
class ItemPriceFactory extends Factory
{
    /**
     * The name of the factory's corresponding model.
     *
     * @var class-string<ItemPrice>
     */
    protected $model = ItemPrice::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'item_id' => Item::factory()->state([
                'type' => fake()->randomElement(ItemType::options()),
            ]),
            'buy_for' => $buyPrice = $this->buyPrice(),
            'buy_currency_id' => $buyCurrency = $this->buyCurrency($buyPrice),

            'sell_for' => $sellPrice = self::sellPrice($buyPrice),
            'sell_currency_id' => $this->sellCurrency($sellPrice, $buyCurrency),
        ];
    }

    protected function buyPrice(): ?float
    {
        return fake()->optional(0.4)->randomFloat(2, 100, 1000);
    }

    protected function buyCurrency(?float $buyPrice): ?int
    {
        return $buyPrice === null
            ? $this->usdCurrency()
            : $this->generateCurrency();
    }

    public static function sellPrice(?float $buyPrice): ?float
    {
        $sellPriceIsNeeded = $buyPrice === null;

        $buyPrice ??= fake()->randomFloat(2, 0, 200);
        $profit = fake()->boolean(40) ? null : fake()->randomFloat(2, 50, 200);

        return !$sellPriceIsNeeded && $profit === null ? null : $buyPrice + $profit;
    }

    protected function sellCurrency(?float $sellPrice, ?int $buyCurrency): ?int
    {
        if ($sellPrice === null) {
            return $this->usdCurrency();
        }

        if ($buyCurrency === null) {
            return $this->generateCurrency();
        }

        return fake()->boolean(90)
            ? $buyCurrency : $this->generateCurrency();
    }

    protected function generateCurrency(): int
    {
        return Currency::factory()->create()->id;
    }

    protected function usdCurrency(): int
    {
        try {
            return Currency::defaultCurrency()->id;
        } catch (ModelNotFoundException) {
            return Currency::factory()->usd()->create()->id;
        }
    }
}
