<?php

declare(strict_types=1);

namespace Bridge\Support\Pagination;

use Bridge\Support\Data\Data;
use Bridge\Support\Data\PrettyDate;
use Bridge\Support\Pagination\Data\PaginatedRecordsData;
use Carbon\CarbonImmutable;
use Closure;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;
use Spatie\LaravelData\Attributes\MapInputName;
use Spatie\LaravelData\Attributes\WithTransformer;
use Spatie\QueryBuilder\QueryBuilder;

/**
 * @template TData of DataWithPaginator
 */
class DataWithPaginator extends Data
{
    public int $maskedId;

    #[MapInputName('created_at')]
    #[WithTransformer(PrettyDate::class)]
    public CarbonImmutable $createdOn;

    /**
     * @param  Closure(QueryBuilder): QueryBuilder  $builder
     * @param  (Closure(TData):TData)|null  $transformer
     * @return array{records: PaginatedRecordsData<TData>}
     */
    public static function paginate(Builder $subject, Closure $builder, ?Closure $transformer = null, ?int $perPage = null): array
    {
        /** @var \Illuminate\Database\Eloquent\Builder<Model> $subjectClone */
        $subjectClone = clone $subject;
        $query = QueryBuilder::for($subjectClone);

        /** @var LengthAwarePaginator<Model> $paginated */
        $paginated = static::collect($builder($query)->paginate($perPage));

        if ($transformer) {
            $paginated->through($transformer);
        }

        /** @var PaginatedRecordsData<TData> $records */
        $records = PaginatedRecordsTransformer::transform($subject, $paginated);

        return compact('records');
    }

    /**
     * @return PaginatedRecordsData<TData>
     */
    public static function paginateRecords(Builder $subject, Closure $builder, ?Closure $transformer = null, ?int $perPage = null): PaginatedRecordsData
    {
        $data = static::paginate($subject, $builder, $transformer, $perPage);

        return $data['records'];
    }
}
