<?php

declare(strict_types=1);

namespace Bridge\Support\Actions;

use Bridge\Support\Actions\Interfaces\InterruptableAction;
use Bridge\Support\MaskedId\MaskedId;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\RedirectResponse;
use Illuminate\Validation\Rule;
use Lorisleiva\Actions\ActionRequest;
use Lorisleiva\Actions\Concerns\AsController;

abstract class AbstractAction
{
    use AsController;

    /**
     * @return class-string<Model>
     */
    abstract public function model(): string;

    abstract public function redirect(): string;

    /**
     * @param  Builder<Model>  $query
     */
    abstract protected function performAction(Builder $query, ActionRequest $request): void;

    /**
     * @return array<string, mixed[]>
     */
    public function rules(): array
    {
        return [
            'records' => ['required', 'array', Rule::exists($this->model(), $this->identifierColumn())],
        ];
    }

    public function handle(ActionRequest $request): RedirectResponse
    {
        $query = $this->query($request);

        // If the action is interruptable, check if the action should be interrupted.
        $interrupted = $this instanceof InterruptableAction
            ? $this->interruptAction($query, $request)
            : false;

        // If the action was not interrupted, perform the action.
        if (!$interrupted) {
            $this->performAction($this->query($request), $request);
        }

        return redirect()->to($this->redirect());
    }

    /**
     * @return Builder<Model>
     */
    protected function query(ActionRequest $request): Builder
    {
        /** @var Model $model */
        $model = $this->model();

        return $model::query()->whereIn(
            column: $this->identifierColumn(),
            values: $request->input('records')
        );
    }

    public function prepareForValidation(ActionRequest $request): void
    {
        if ($this->identifierColumn() !== 'id') {
            return;
        }

        $request->merge([
            'records' => app(MaskedId::class)->decodeModelIds($request->input('records')),
        ]);
    }

    protected function identifierColumn(): string
    {
        return 'id';
    }
}
