<?php

declare(strict_types=1);

namespace Bridge;

use Bridge\Domains\Invoice\Enums\InvoiceContactType;
use Bridge\Domains\Invoice\Enums\InvoiceGeneratorType;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Member\Customer;
use Bridge\Domains\Server\Models\Server;
use Bridge\Domains\Server\Support\CredentialEncrypter;
use Bridge\Domains\Supplier\Supplier;
use Bridge\Support\MaskedId\MaskedId;
use Bridge\Support\Toast\Toast;
use Carbon\CarbonImmutable;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\ServiceProvider;
use Jenssegers\Optimus\Optimus;
use Spatie\Activitylog\Facades\CauserResolver;

class PackageServiceProvider extends ServiceProvider
{
    public function boot(): void
    {
        $this->loadViewsFrom(__DIR__.'/../resources/views', 'bridge');

        Relation::morphMap([
            // Invoice contact types
            InvoiceContactType::Supplier->value => Supplier::class,
            InvoiceContactType::Customer->value => Customer::class,

            // Invoice generator types
            InvoiceGeneratorType::Server->value => Server::class,
            InvoiceGeneratorType::Invoice->value => Invoice::class,
        ]);

        Date::use(CarbonImmutable::class);
        $formats = config('data.date_format');
        config()->set('data.date_format', [
            'Y-m-d',
            'Y-m-d H:i:s',
            // ISO 8601, laravel model toArray() uses this format
            'Y-m-d\TH:i:s.uT',
            ...Arr::wrap($formats),
        ]);

        $this->loadMigrationsFrom(__DIR__.'/../database/migrations');

        if (!empty(config('app.key')) && Auth::check()) {
            CauserResolver::setCauser(Auth::user());
        }
    }

    public function register(): void
    {
        $this->app->singleton(Toast::class, fn () => new Toast());

        $this->app->singleton(
            CredentialEncrypter::class,
            fn () => new CredentialEncrypter(config('services.encryption.credential_key'))
        );

        $this->app->singleton(MaskedId::class, fn () => new MaskedId(
            optimus: new Optimus(
                (int) config('services.optimus.prime'),
                (int) config('services.optimus.inverse'),
                (int) config('services.optimus.random'),
                size: 31
            )
        ));
    }
}
