<?php

declare(strict_types=1);

namespace Bridge\Domains\Ticket\Models;

use Bridge\Database\Factories\TicketFactory;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Member\Customer;
use Bridge\Domains\Member\Enums\MemberType;
use Bridge\Domains\Notification\HasNotifications;
use Bridge\Domains\Ticket\Enums\TicketCreator;
use Bridge\Domains\Ticket\Enums\TicketStatus;
use Bridge\Domains\Ticket\Support\HasTicketNumber;
use Bridge\Support\ModelTraits;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @property-read Customer $customer
 */
class Ticket extends Model
{
    use HasFactory;
    use HasNotifications;
    use HasTicketNumber;
    use ModelTraits;
    use SoftDeletes;

    protected $casts = [
        'status' => TicketStatus::class,
        'created_by' => TicketCreator::class,
        'last_reply_from' => MemberType::class,
    ];

    /**
     * @return BelongsTo<Customer,Ticket>
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * @return BelongsTo<Invoice,Ticket>
     */
    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * @return HasMany<TicketReply>
     */
    public function replies(): HasMany
    {
        return $this->hasMany(TicketReply::class);
    }

    /**
     * @return Factory<Ticket>
     */
    protected static function newFactory(): Factory
    {
        return TicketFactory::new();
    }
}
