<?php

declare(strict_types=1);

namespace Bridge\Domains\Notification\Customer\Invoice;

use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Member\Customer;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Address;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Number;

class InvoiceCreated extends Mailable
{
    use Queueable, SerializesModels;

    public function __construct(public Invoice $record)
    {
        $this->afterCommit();
        $this->record->loadMissing(['contact', 'currency', 'items', 'items.item']);
    }

    public function envelope(): Envelope
    {
        return new Envelope(
            to: [
                new Address(
                    address: $this->customer()->email,
                    name: $this->customer()->name,
                ),
            ],
            subject: "[#{$this->record->number}] Invoice Generated",
        );
    }

    public function content(): Content
    {
        return new Content(
            markdown: 'bridge::mails.customer.invoice.created',
            with: [
                'customer' => $this->customer(),
                'supportUrl' => route('tickets.new'),
                'invoiceItems' => $this->formattedItems(),
                'formattedTotal' => $this->formattedTotal(),
                'invoiceViewUrl' => route('invoices.index'),
            ]
        );
    }

    private function formattedItems(): string
    {
        return $this->record->items
            ->map(fn ($item) => "{$item->quantity}x {$item->item->name}")
            ->implode(', ');
    }

    protected function formattedTotal(): string
    {
        $amount = $this->record->total;
        $symbol = $this->record->currency->symbol;

        return $symbol.Number::format($amount, precision: 2, locale: 'en-IN');
    }

    protected function customer(): Customer
    {
        return $this->record->contact;
    }
}
