<?php

declare(strict_types=1);

namespace Bridge\Domains\Member;

use Bridge\Database\Factories\MemberFactory;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Member\Enums\MemberType;
use Bridge\Domains\Member\Models\MemberItem;
use Bridge\Domains\Member\Support\HasAvatar;
use Bridge\Domains\Ticket\Models\Ticket;
use Bridge\Support\ModelTraits;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User;
use Illuminate\Notifications\Notifiable;
use Laravel\Fortify\TwoFactorAuthenticatable;

class Member extends User implements MustVerifyEmail
{
    use HasAvatar;
    use HasFactory;
    use ModelTraits;
    use Notifiable;
    use SoftDeletes;
    use TwoFactorAuthenticatable;

    protected $table = 'members';

    protected $hidden = [
        'password',
        'remember_token',
        'email_verified_at',
        'two_factor_recovery_codes',
        'two_factor_secret',
        'two_factor_confirmed_at',
    ];

    protected $casts = [
        'password' => 'hashed',
        'type' => MemberType::class,
        'email_verified_at' => 'datetime',
    ];

    /**
     * @return Attribute<string, never>
     */
    protected function moniker(): Attribute
    {
        return Attribute::get(fn () => $this->nickname ?? $this->name);
    }

    /**
     * @param  Builder<Member>  $query
     */
    public function scopeIsStaff(Builder $query): void
    {
        $query->where('type', MemberType::Staff);
    }

    /**
     * @param  Builder<Member>  $query
     */
    public function scopeIsCustomer(Builder $query): void
    {
        $query->where('type', MemberType::Customer);
    }

    /**
     * @return MorphMany<Invoice>
     */
    public function invoices(): MorphMany
    {
        return $this->morphMany(Invoice::class, 'contact');
    }

    /**
     * @return HasMany<Ticket>
     */
    public function tickets(): HasMany
    {
        return $this->hasMany(Ticket::class, 'customer_id');
    }

    /**
     * @return HasMany<MemberItem>
     */
    public function items(): HasMany
    {
        return $this->hasMany(MemberItem::class, 'member_id');
    }

    public static function loggedIn(): static
    {
        /** @var static $logged */
        $logged = auth()->user();

        return $logged;
    }

    /**
     * @return Factory<Member>
     */
    protected static function newFactory(): Factory
    {
        return MemberFactory::new();
    }
}
