<?php

declare(strict_types=1);

namespace Bridge\Domains\License\Models;

use Bridge\Domains\License\Enums\LicenseStatus;
use Bridge\Domains\Member\Customer;
use Bridge\Domains\Product\ProductType;
use Bridge\Domains\Supplier\Supplier;
use Bridge\Support\ModelTraits;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasOneThrough;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @property LicenseStatus $status
 * @property-read Supplier $customer
 * @property-read LicenseProduct $product
 * @property-read Product $coreProduct
 */
class License extends Model
{
    use ModelTraits;
    use SoftDeletes;

    protected $connection = 'license_mysql';

    protected $casts = [
        'status' => LicenseStatus::class,
    ];

    public const int FREE_UPDATE_MONTHS = 6;

    /**
     * @return BelongsTo<Customer,License>
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * @return HasMany<LicenseProduct>
     */
    public function licensed(): HasMany
    {
        return $this->hasMany(LicenseProduct::class);
    }

    /**
     * @return HasMany<LicensePing>
     */
    public function pings(): HasMany
    {
        return $this->hasMany(LicensePing::class);
    }

    /**
     * @return HasOne<LicenseProduct>
     */
    public function product(): HasOne
    {
        return $this->hasOne(LicenseProduct::class)
            ->whereHas('product', fn ($query) => $query->where('type', ProductType::Core));
    }

    /**
     * @return HasMany<LicenseProduct>
     */
    public function addons(): HasMany
    {
        return $this->hasMany(LicenseProduct::class)
            ->whereHas('product', fn ($query) => $query->where('type', ProductType::Addon));
    }

    /**
     * @return HasOneThrough<Product>
     */
    public function coreProduct(): HasOneThrough
    {
        return $this->hasOneThrough(Product::class, LicenseProduct::class, 'license_id', 'id', 'id', 'product_id')
            ->where('type', ProductType::Core);
    }

    /**
     * @param  Builder<static>  $query
     * @return Builder<static>
     */
    public function scopeAlive(Builder $query): Builder
    {
        return $query->whereHas('pings', fn (Builder $query) => $query->where('created_at', '>=', now()->subDays(7)));
    }

    public function isAlive(): bool
    {
        return $this->pings()->where('created_at', '>=', now()->subDays(7))->exists();
    }

    protected function newRelatedInstance($class): Model
    {
        /** @var Model $instance */
        $instance = new $class;

        if (!$instance->getConnectionName()) {
            $instance->setConnection(config('database.default'));
        }

        return $instance;
    }
}
