<?php

declare(strict_types=1);

namespace Bridge\Domains\Invoice\Tasks;

use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Invoice\Models\InvoiceItem;
use Bridge\Domains\License\Models\License;
use Bridge\Domains\License\Models\LicenseProduct;
use Bridge\Domains\License\Models\Product;
use Illuminate\Support\Facades\DB;

abstract class AbstractTask
{
    abstract protected static function handle(Invoice $invoice): void;

    public static function run(Invoice $invoice): void
    {
        if (!$invoice->type->isRecurring()) {
            DB::transaction(fn () => static::handle($invoice));
        }
    }

    protected static function cleanupPayments(Invoice $invoice): void
    {
        $invoice->payments()->delete();
    }

    protected static function cleanupItemOwnership(Invoice $invoice): void
    {
        self::getTargetInvoice($invoice)->memberItems()->delete();
    }

    protected static function cleanupItemLicenses(Invoice $invoice): void
    {
        // Don't do anything if the invoice was generated by a recurring invoice
        if (self::getTargetInvoice($invoice)->type->isRecurring() || app()->runningUnitTests()) {
            return;
        }

        self::cleanupLicenseCoreItem($invoice);
        self::cleanupLicenseAddonItems($invoice);
    }

    private static function cleanupLicenseCoreItem(Invoice $invoice): void
    {
        $query = License::query()
            ->whereHas('coreProduct', function ($query) use ($invoice) {
                return $query->where('invoice_id', $invoice->id);
            });

        $query->update(['message' => 'Auto deleted due to invoice status change: '.$invoice->status->value]);
        $query->delete();
    }

    private static function cleanupLicenseAddonItems(Invoice $invoice): void
    {
        $localAddons = $invoice->items->map(fn (InvoiceItem $item) => $item->item_id);
        $licenseAddons = Product::query()->whereIn('erp_product_id', $localAddons)->pluck('id');

        LicenseProduct::query()
            ->whereIn('product_id', $licenseAddons)
            ->where('invoice_id', $invoice->id)
            ->delete();
    }

    protected static function getTargetInvoice(Invoice $invoice): Invoice
    {
        if ($invoice->generator instanceof Invoice && $invoice->generator->type->isRecurring()) {
            return $invoice->generator;
        }

        return $invoice;
    }
}
