<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Member\Customer;
use Bridge\Domains\Member\Member;
use Bridge\Domains\Server\Data\ServerCredentialData;
use Bridge\Domains\Server\Enums\ServerLoginMethod;
use Bridge\Domains\Server\Models\ItemServerDetails;
use Bridge\Domains\Server\Models\Server;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Collection;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<ItemServerDetails>
 */
class ItemServerDetailsFactory extends Factory
{
    protected $model = ItemServerDetails::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'item_id' => Server::factory(),
            'customer_id' => $this->getAssignee(),
            'ip_address' => fake()->unique()->ipv4,
            'port' => fake()->numberBetween(21, 65535),
            'due_within' => fake()->randomElement([0, 3, 7, 10, 15, 30]),
            'invoice_date' => fake()->dateTimeBetween('-1 week')->format('Y-m-d'),
            'credentials' => ServerCredentialData::collect($this->randomCredentials()),
        ];
    }

    /**
     * @return Collection<int,ServerCredentialData>
     */
    protected function randomCredentials(): Collection
    {
        return Collection::times(rand(1, 2), fn () => new ServerCredentialData(
            $method = fake()->randomElement(ServerLoginMethod::cases()),
            username: $this->randomPossibleUsername(),
            password: $method === ServerLoginMethod::Key ? null : fake()->password,
        ));
    }

    protected function randomPossibleUsername(): string
    {
        return fake()->randomElement(['root', 'ploi', 'www', 'admin', 'ftp', 'user', 'ubuntu', 'centos', 'debian']);
    }

    /**
     * @return Factory<Member>|null
     */
    private function getAssignee(): ?Factory
    {
        return mt_rand(1, 10) > 8 ? null : Customer::factory();
    }
}
