<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Item\Enums\ItemStatus;
use Bridge\Domains\Item\Enums\ItemType;
use Bridge\Domains\Item\Models\Item;
use Bridge\Domains\Item\Models\ItemMeta;
use Bridge\Domains\Item\Models\ItemPrice;
use Bridge\Domains\Supplier\Supplier;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * @template TModel of Item
 *
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<TModel>
 */
class ItemFactory extends Factory
{
    protected $model = Item::class;

    /**
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'supplier_id' => Supplier::factory(),
            'name' => rtrim(fake()->text(rand(20, 40)), '.'),
            'type' => fake()->randomElement(ItemType::options()),
            'status' => fake()->randomElement(ItemStatus::options()),
        ];
    }

    public function configure(): static
    {
        return $this->afterCreating(function (Model $item) {
            // Set price
            ItemPrice::factory()->for($item, 'item')->create();

            // Insert meta data
            foreach ($this->metaData() as $key => $value) {
                ItemMeta::factory()->for($item, 'item')->state([
                    'key' => $key,
                    'value' => $value,
                ])->create();
            }
        });
    }

    /**
     * @return array<string, int|string>
     */
    public function metaData(): array
    {
        return [
            Str::ulid()->toBase32() => fake()->word(),
        ];
    }
}
