<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Invoice\Enums\RecurringEndType;
use Bridge\Domains\Invoice\Enums\RecurringFrequency;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Invoice\Models\InvoiceSchedule;
use Closure;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<InvoiceSchedule>
 */
class InvoiceScheduleFactory extends Factory
{
    /**
     * The name of the factory's corresponding model.
     *
     * @var class-string<InvoiceSchedule>
     */
    protected $model = InvoiceSchedule::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'invoice_id' => Invoice::factory()->recurring(),
            'frequency' => fake()->randomElement(RecurringFrequency::options()),
            'interval' => fake()->randomElement([1, 3, 6, 12]),
            'due_within' => fake()->randomElement([0, 3, 7, 10, 15, 30]),
            'start_on' => fake()->dateTimeBetween('-1 week')->format('Y-m-d'),
            'end_type' => fake()->randomElement(RecurringEndType::cases()),

            'end_on' => $this->generateEndOn(),
            'end_after' => $this->generateEndAfter(),
        ];
    }

    private function generateEndOn(): Closure
    {
        return function (array $attributes) {
            if ($attributes['end_type'] !== RecurringEndType::OnDate) {
                return null;
            }

            return fake()->dateTimeBetween($attributes['start_on'], '+1 year')->format('Y-m-d');
        };
    }

    private function generateEndAfter(): Closure
    {
        return function (array $attributes) {
            if ($attributes['end_type'] !== RecurringEndType::AfterCount) {
                return null;
            }

            return rand(5, 20);
        };
    }
}
